/*
 * Copyright 2011 Tilera Corporation. All Rights Reserved.
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation, version 2.
 *
 *   This program is distributed in the hope that it will be useful, but
 *   WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, GOOD TITLE or
 *   NON INFRINGEMENT.  See the GNU General Public License for
 *   more details.
 */

/*
 * This is an IDE implementation for the Tile processor, using the
 * GPIO pins provided by the Tile without any external controller.
 */

#include <linux/ide.h>
#include <linux/ioport.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/platform_device.h>
#include <linux/irq.h>
#include <linux/string.h>
#include <linux/delay.h>
#include <linux/io.h>

#include <asm/hv_driver.h>

#include <hv/drv_gpio_intf.h>

#define SET_BIT(x)  (1<<(x))
#define CLR_BIT(x)  (0<<(x))
#define MASK_BIT(x) (1<<(x))

/*
 * Some comments on the signals:
 *
 * WE_L should always be HIGH in TrueIDE mode.
 * ATA_SEL_L should be pulled LOW to enable TrueIDE mode.
 * DMACK_L should be pulled high to indicate that DMA is not supported.
 * DMARQ is not used when DMA is not supported and should be tristated.
 * PDIAG should be tristated.
 * DASP should be tristated.
 * CD1 and CD2 are card detect, and are used to detect that a CF/CF+
 *   card is fully inserted.
 * INTRQ is the active high interrupt request to host.
 * IORDY (WAIT) - not ever used by some devices.
 */

/*
 * cf_ctl1_bank is used for:
 *  INTRQ
 *  RESET
 *  PDIAG_L
 *  DASP_L
 * These signals are only used during device initialization.
 */

#define CF_INTRQ_BIT		1
#define CF_RESET_BIT		2
#define CF_PDIAG_L_BIT		5
#define CF_DASP_L_BIT		6

/* This is the mask for all the bits on CTL1 that we use. */
#define CF_CTL1_MASK			  \
	(MASK_BIT(CF_INTRQ_BIT)		| \
	 MASK_BIT(CF_RESET_BIT)		| \
	 MASK_BIT(CF_PDIAG_L_BIT)	| \
	 MASK_BIT(CF_DASP_L_BIT))

/*
 * cf_ctl2_bank is used for:
 *  CS0_L
 *  CS1_L
 *  WE_L
 *  ATA_SEL_L
 *  DMACK_L
 *  IOCS16_L
 *  IORD_L
 *  IOWR_L
 *  DMARQ_L
 *  IORDY_L
 *  ADDR[0:2]
 */

#define CF_CS1_L_BIT		0
#define CF_CS0_L_BIT		1
#define CF_WE_L_BIT		2
#define CF_ATA_SEL_L_BIT	3
#define CF_DMACK_L_BIT		4
#define CF_IOCS16_L_BIT		9
#define CF_IORD_L_BIT		10
#define CF_IOWR_L_BIT		11
#define CF_DMARQ_BIT		12
#define CF_IORDY_BIT		13

#define CF_ADDR_BIT		5
#define CF_ADDR_WIDTH		3
#define CF_ADDR_MASK (((1 << (CF_ADDR_WIDTH)) - 1) << CF_ADDR_BIT)

#define CF_DATA_BIT		0
#define CF_DATA_WIDTH		16

/* This masks out the DMA-related bits on CTL2. */
#define CF_CTL2_NODMA_MASK            \
	(~(MASK_BIT(CF_DMACK_L_BIT) | \
	   MASK_BIT(CF_DMARQ_BIT)))

/* This is the mask for all the bits on CTL2 that we use. */
#define CF_CTL2_MASK		      \
	(MASK_BIT(CF_CS0_L_BIT)	    | \
	 MASK_BIT(CF_CS1_L_BIT)	    | \
	 MASK_BIT(CF_DMACK_L_BIT)   | \
	 MASK_BIT(CF_IORD_L_BIT)    | \
	 MASK_BIT(CF_IOWR_L_BIT)    | \
	 CF_ADDR_MASK)

/* Non-DMA version of all of the bits we use. */
#define CF_CTL2_MASK_NODMA            \
	(CF_CTL2_MASK & CF_CTL2_NODMA_MASK)

/* These are the values for the bits on CTL2 that don't change in operation. */
#define CF_CTL2_CONST_BITS	     \
	 SET_BIT(CF_DMACK_L_BIT)

/* Non-DMA version of constant bits. */
#define CF_CTL2_CONST_BITS_NODMA     \
	(CF_CTL2_CONST_BITS & CF_CTL2_NODMA_MASK)

/* These are the bit settings that I want leading up to a read or write. */
#define CF_CTL2_RW_SETUP_BITS	  \
	(CF_CTL2_CONST_BITS	| \
	 SET_BIT(CF_CS1_L_BIT)	| \
	 CLR_BIT(CF_CS0_L_BIT)	| \
	 SET_BIT(CF_IOWR_L_BIT) | \
	 SET_BIT(CF_IORD_L_BIT))

/* Non-DMA version of RW setup bits. */
#define CF_CTL2_RW_SETUP_BITS_NODMA  \
	(CF_CTL2_RW_SETUP_BITS & CF_CTL2_NODMA_MASK)

/* These are the bit settings that I want when I'm done with an operation. */
#define CF_CTL2_DEFAULT_BITS	  \
	(CF_CTL2_CONST_BITS	| \
	 SET_BIT(CF_CS1_L_BIT)	| \
	 CLR_BIT(CF_CS0_L_BIT)	| \
	 SET_BIT(CF_IOWR_L_BIT) | \
	 SET_BIT(CF_IORD_L_BIT))

/* Non-DMA version of default bits. */
#define CF_CTL2_DEFAULT_BITS_NODMA  \
	(CF_CTL2_DEFAULT_BITS & CF_CTL2_NODMA_MASK)

/*
 * Defines for where in the little programs register
 * space constants and other useful things live.
 */
#define CF_VM_CONST_FFFFFFFF	255
#define CF_VM_CONST_PARAM_0	254
#define CF_VM_CONST_PARAM_1	253
#define CF_VM_CONST_PARAM_2	252
#define CF_VM_CONST_PARAM_3	251
#define CF_VM_CONST_PARAM_4	250
#define CF_VM_CONST_IORD_L_BIT	249
#define CF_VM_CONST_DONE	248
#define CF_VM_CONST_IOWR_L_BIT	247
#define CF_VM_CONST_IORDY_BIT	246
#define CF_VM_CONST_CTL2_MASK	245
#define CF_VM_CONST_0		0
#define CF_VM_CONST_1		1
#define CF_VM_CONST_2		2
#define CF_VM_CONST_99		99
#define CF_VM_REG_0		100
#define CF_VM_REG_1		101
#define CF_VM_REG_2		102
#define CF_VM_REG_3		103
#define CF_VM_REG_4		104
#define CF_VM_REG_5		105
#define CF_VM_REG_6		106
#define CF_VM_REG_7		107

#define TILE_IDE_MAX_TRANSFER_BYTE_SIZE 1012

#define DRV_NAME "ide-tile"

#undef _TIDE_DEBUG
#ifdef _TIDE_DEBUG
#define TIDE_DEBUG printk
#else
#define TIDE_DEBUG(...)
#endif

#define TILE_IDE_NO_WAIT

/*
 * There are two ways we can run.  Without the Hypervisor programs,
 * the operations are done from this driver with discrete calls into
 * the Hypervisor for each bit operation.
 *
 * With the Hypervisor programs, the functionality for each operation
 * is downloaded into the GPIO driver in the Hypervisor, resulting in
 * much faster operation due to fewer calls into the Hypervisor.
 *
 * The functionality should be identical in each case, the only
 * difference is speed.
 */
#define TIDE_USE_HYPERVISOR_PROGRAMS

struct tide_s {
	int dev_fds;
};

static struct tide_s tide_satchel;

static int cf_ctl1_bank;
static int cf_ctl2_bank;
static int cf_data_bank;

static int cf_has_dma_bits = 1;

static int cf_ctl2_mask = CF_CTL2_MASK;
static int cf_ctl2_const_bits = CF_CTL2_CONST_BITS;
static int cf_ctl2_rw_setup_bits = CF_CTL2_RW_SETUP_BITS;
static int cf_ctl2_default_bits = CF_CTL2_DEFAULT_BITS;

/**
 * Set GPIO pins to particular value
 *
 * @param bits Sixteen bits to set
 * @param mask Which of the bits 16 bits to set
 * @param bank The bank to operate on
 */
static int tile_ide_set_bits(u32 bits, u32 mask, int bank)
{
	u32 buffer[2];
	u32 error;

	buffer[0] = bits;
	buffer[1] = mask;
	error = hv_dev_pwrite(tide_satchel.dev_fds, 0,
			      (HV_VirtAddr)&buffer[0],
			      8,
			      GPIO_CTL_CONFIG_DATA_OFF +
			      (2 * sizeof(u32) * bank));
	if (error != 8) {
		pr_err(DRV_NAME ":HV GPIO problem: devd: %d "
		       "error: %d set bits bits: 0x%x mask: 0x%x bank: %d\n",
		       tide_satchel.dev_fds, error, bits, mask, bank);
		return -1;
	}
	return 0;
}

/**
 * Turn denoted bits into inputs
 *
 * @param buffer One-hot mask tells which bits to release
 * @param bank The bank to operate on
 */
static int tile_ide_release_bits(u32 buffer, int bank)
{
	if (hv_dev_pwrite(tide_satchel.dev_fds, 0, (HV_VirtAddr)&buffer,
			  4,
			  GPIO_CTL_CONFIG_DIR_OFF + (sizeof(u32) * bank)) !=
	    4) {
		pr_err(DRV_NAME ":HV GPIO problem: "
		       "release bits bits: 0x%x bank: %d\n", buffer, bank);
		return -1;
	}
	return 0;
}

/**
 * Read the bits from the noted bank
 *
 * @param bank The bank to operate on
 * @result Current state of the bits
 */
static u32 tile_ide_read_bits(int bank)
{
	u32 buffer;
	if (hv_dev_pread(tide_satchel.dev_fds, 0, (HV_VirtAddr)&buffer,
			 4, GPIO_CTL_CONFIG_DATA_OFF + (2 * sizeof(u32) * bank))
	    != 4) {
		pr_err(DRV_NAME ":HV GPIO problem: "
		       "read bits bank: %d\n", bank);
	}
	return buffer;
}

#ifdef TIDE_USE_HYPERVISOR_PROGRAMS
/**
 * Write a little program instruction to the gpio hv driver
 *
 * @param gpio_mem_location Address to write instruction in hv space
 * @param opcode Opcode for the instruction
 * @param dest_operand Destination register
 * @param src_a_operand First source regsiter
 * @param src_b_operand Second source regsiter
 * @result address of next instruction
 */
static u32 tile_ide_write_instruction(u32 gpio_mem_location, u8 opcode,
			   u8 dest_operand, u8 src_a_operand,
			   u8 src_b_operand)
{
	gpio_instruction_t instruction;
	instruction.opcode = opcode;
	instruction.dest_operand = dest_operand;
	instruction.src_a_operand = src_a_operand;
	instruction.src_b_operand = src_b_operand;
	hv_dev_pwrite(tide_satchel.dev_fds, 0, (HV_VirtAddr)&instruction,
		      sizeof(gpio_instruction_t), gpio_mem_location);
	return gpio_mem_location + sizeof(gpio_instruction_t);
}

/**
 * Write a little program immediate instruction to the gpio hv driver
 *
 * @param gpio_mem_location Address to write instruction in hv space
 * @param opcode Opcode for the instruction
 * @param dest_operand Destination register
 * @param immediate The instruction immediate
 * @result address of next instruction
 */
static u32 tile_ide_write_immediate_instruction(u32 gpio_mem_location,
				     u8 opcode, u8 dest_operand,
				     u16 immediate)
{
	gpio_instruction_t instruction;
	instruction.opcode = opcode;
	instruction.dest_operand = dest_operand;
	instruction.immediate = immediate;
	hv_dev_pwrite(tide_satchel.dev_fds, 0, (HV_VirtAddr)&instruction,
		      sizeof(gpio_instruction_t), gpio_mem_location);
	return gpio_mem_location + sizeof(gpio_instruction_t);
}

static u32 tile_ide_inl_program;

static u32 tile_ide_inl(unsigned long port)
{
	u32 buffer[2];

	buffer[0] = tile_ide_inl_program;	/* The program to run. */
	/* do a little bit of processing here */
	buffer[1] =
		((port & 0x7) << CF_ADDR_BIT) |
		CLR_BIT(CF_CS0_L_BIT)	      |
		SET_BIT(CF_IORD_L_BIT)	      |
		SET_BIT(CF_IOWR_L_BIT);
	/* Writing to GPIO_CTL_WRITE_OFF starts the program. */
	if (hv_dev_pwrite(tide_satchel.dev_fds, 0, (HV_VirtAddr)&buffer[0],
			  8, GPIO_CTL_WRITE_OFF) != 8) {
		pr_err(DRV_NAME ":HV GPIO problem: "
		       "inl write program problem\n");
	}
	if (hv_dev_pread(tide_satchel.dev_fds, 0, (HV_VirtAddr)&buffer[0],
			 4, GPIO_CTL_CONFIG_RF_OFF + (CF_VM_REG_0 * 4)) != 4) {
		pr_err(DRV_NAME ":HV GPIO problem: "
		       "inl read program problem\n");
	}
	TIDE_DEBUG(DRV_NAME ":  inl port: 0x%03x data: 0x%04x\n",
		   port, buffer[0]);
	return buffer[0];
}

static u32 tile_ide_compose_inl_program(u32 gpio_program_address)
{
#ifndef TILE_IDE_NO_WAIT
	u32 target_3;
#endif
	tile_ide_inl_program = gpio_program_address;

	/* Put data bus into read mode. */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_RELEASE_PINS,
				       cf_data_bank,
				       CF_VM_CONST_FFFFFFFF,
				       CF_VM_CONST_0);

	/* Load the parameter passed in into CF_VM_REG_0. */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_LW,
				       CF_VM_REG_0,
				       CF_VM_CONST_PARAM_0,
				       CF_VM_CONST_0);

	/* This sets the bits passed in, which are now in CF_VM_REG_0. */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_SET_PINS,
				       cf_ctl2_bank,
				       CF_VM_REG_0,
				       CF_VM_CONST_CTL2_MASK);

	/* Assert IORD_L (set it to 0). */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_SET_PINS,
				       cf_ctl2_bank,
				       CF_VM_CONST_0,
				       CF_VM_CONST_IORD_L_BIT);

#ifndef TILE_IDE_NO_WAIT
	/* read the wait bit/bank */
	target_3 = gpio_program_address;
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_READ_PINS,
				       CF_VM_REG_0,
				       cf_ctl2_bank,
				       0);
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_AND,
				       CF_VM_REG_0,
				       CF_VM_REG_0,
				       CF_VM_CONST_IORDY_BIT);
	gpio_program_address =
	    tile_ide_write_immediate_instruction(gpio_program_address,
						 GPIO_BZ,
						 CF_VM_REG_0,
						 (target_3 - program_address) /
						 sizeof(gpio_instruction_t));
#endif

	/* Read the data. */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_READ_PINS,
				       CF_VM_REG_0,
				       cf_data_bank,
				       0);

	/* Deassert IORD_L (set it to 1). */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_SET_PINS,
				       cf_ctl2_bank,
				       CF_VM_CONST_IORD_L_BIT,
				       CF_VM_CONST_IORD_L_BIT);

	/* Put the control pins back to their resting values. */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_SET_PINS,
				       cf_ctl2_bank,
				       CF_VM_CONST_DONE,
				       CF_VM_CONST_CTL2_MASK);

	/* And we're done. */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_DONE,
				       CF_VM_REG_0,
				       CF_VM_REG_0,
				       CF_VM_REG_0);
	return gpio_program_address;
}

static u32 tile_ide_insw_program;

static u32 tile_ide_compose_insw_program(u32 gpio_program_address)
{
	u32 target_1;
#ifndef TILE_IDE_NO_WAIT
	u32 target_3;
#endif
	tile_ide_insw_program = gpio_program_address;

	/* Put data bus into read mode. */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_RELEASE_PINS,
				       cf_data_bank,
				       CF_VM_CONST_FFFFFFFF,
				       CF_VM_CONST_0);

	/* Load the parameter passed in into CF_VM_REG_0. */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_LW,
				       CF_VM_REG_0,
				       CF_VM_CONST_PARAM_0,
				       CF_VM_CONST_0);

	/* This sets the bits passed in, which are now in CF_VM_REG_0. */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_SET_PINS,
				       cf_ctl2_bank,
				       CF_VM_REG_0,
				       CF_VM_CONST_CTL2_MASK);

	/* Load our loop counter with number of transactions to do. */
	/* CF_VM_REG_1 contains the read length. */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_LW,
				       CF_VM_REG_1,
				       CF_VM_CONST_PARAM_1,
				       CF_VM_CONST_0);

	/* CF_VM_REG_2 contains the address. */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_ADD,
				       CF_VM_REG_2,
				       CF_VM_CONST_0,
				       CF_VM_CONST_0);

	gpio_program_address =
	    tile_ide_write_immediate_instruction(gpio_program_address,
						 GPIO_LI,
						 CF_VM_REG_2,
						 GPIO_CTL_READ_OFF);

	/* Assert IORD_L (set it to 0). */
	target_1 = gpio_program_address;
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_SET_PINS,
				       cf_ctl2_bank,
				       CF_VM_CONST_0,
				       CF_VM_CONST_IORD_L_BIT);

#ifndef TILE_IDE_NO_WAIT
	/* read the wait bit/bank */
	target_3 = gpio_program_address;
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_READ_PINS,
				       CF_VM_REG_0,
				       cf_ctl2_bank,
				       0);
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_AND,
				       CF_VM_REG_0,
				       CF_VM_REG_0,
				       CF_VM_CONST_IORDY_BIT);
	gpio_program_address =
	    tile_ide_write_immediate_instruction(gpio_program_address,
						 GPIO_BZ,
						 CF_VM_REG_0,
						 (target_3 -
						  gpio_program_address) /
						 sizeof(gpio_instruction_t));
#endif

	/* Read the 2 bytes of data into CF_VM_REG_7. */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_READ_PINS,
				       CF_VM_REG_7,
				       cf_data_bank,
				       0);
	/* Deassert IORD_L (set it to 0). */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_SET_PINS,
				       cf_ctl2_bank,
				       CF_VM_CONST_IORD_L_BIT,
				       CF_VM_CONST_IORD_L_BIT);

	/* Store the data at the current address. */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_SH,
				       CF_VM_REG_2,
				       CF_VM_REG_7,
				       CF_VM_CONST_0);

	/* Decrement the counter in CF_VM_REG_1. */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_ADD,
				       CF_VM_REG_1,
				       CF_VM_REG_1,
				       CF_VM_CONST_FFFFFFFF);

	/* Increment the address in CF_VM_REG_2. */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_ADD,
				       CF_VM_REG_2,
				       CF_VM_REG_2,
				       CF_VM_CONST_2);

	/* Branch to target_1, unless we have hit zero. */
	gpio_program_address =
	    tile_ide_write_immediate_instruction(gpio_program_address,
						 GPIO_BNZ,
						 CF_VM_REG_1,
						 (target_1 -
						  gpio_program_address) /
						 sizeof(gpio_instruction_t));

	/* Put the control pins back to their resting values. */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_SET_PINS,
				       cf_ctl2_bank,
				       CF_VM_CONST_DONE,
				       CF_VM_CONST_CTL2_MASK);

	/* And we're done. */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_DONE,
				       CF_VM_REG_0,
				       CF_VM_REG_0,
				       CF_VM_REG_0);
	return gpio_program_address;
}

static void tile_ide_insw(unsigned long port, void *addr, u32 count)
{
	u16 *p = addr;
	u32 transaction_size;
	u32 buffer[3];

	buffer[0] = tile_ide_insw_program;
	buffer[1] =
		((port & 0x7) << CF_ADDR_BIT) |
		CLR_BIT(CF_CS0_L_BIT)	      |
		SET_BIT(CF_IORD_L_BIT)	      |
		SET_BIT(CF_IOWR_L_BIT);
	TIDE_DEBUG(DRV_NAME ": insw port: 0x%x addr: 0x%x count %d\n",
		   port, addr, count);
	while (count) {
		transaction_size = (count > (TILE_IDE_MAX_TRANSFER_BYTE_SIZE /
					     sizeof(u16)))
		    ? (TILE_IDE_MAX_TRANSFER_BYTE_SIZE / sizeof(u16)) : count;
		buffer[2] = transaction_size;
		if (hv_dev_pwrite(tide_satchel.dev_fds, 0,
				  (HV_VirtAddr)&buffer[0], 12,
				  GPIO_CTL_WRITE_OFF) != 12) {
			pr_err(DRV_NAME ":HV GPIO problem: "
			       "insw write program problem\n");
		}
		if (hv_dev_pread(tide_satchel.dev_fds, 0, (HV_VirtAddr)p,
				 transaction_size * sizeof(u16),
				 GPIO_CTL_READ_OFF)
		    != (transaction_size * sizeof(u16))) {
			pr_err(DRV_NAME ":HV GPIO problem: "
			       "insw read program problem\n");
		}
		p += transaction_size;
		count -= transaction_size;
	}
}

static u32 tile_ide_outl_program;

static void tile_ide_outl(u32 val, unsigned long port)
{
	u32 buffer[3];

	/* buffer[0] is the program to run. */
	buffer[0] = tile_ide_outl_program;

	/* buffer[1] is referenced in the program as CF_VM_CONST_PARAM_0. */
	buffer[1] =
		((port & 0x7) << CF_ADDR_BIT) |
		CLR_BIT(CF_CS0_L_BIT)	      |
		SET_BIT(CF_IORD_L_BIT)	      |
		SET_BIT(CF_IOWR_L_BIT);

	/* buffer[2] is referenced in the program as CF_VM_CONST_PARAM_1. */
	buffer[2] = val;

	/* Writing to GPIO_CTL_WRITE_OFF starts the program. */
	if (hv_dev_pwrite(tide_satchel.dev_fds, 0, (HV_VirtAddr)&buffer[0],
			  12, GPIO_CTL_WRITE_OFF) != 12) {
		pr_err(DRV_NAME ":HV GPIO problem: outl write "
		       "program problem\n");
	}
	TIDE_DEBUG(DRV_NAME ": outl port: 0x%03x data: 0x%04x\n", port, val);
}

static u32 tile_ide_compose_outl_program(u32 gpio_program_address)
{
#ifndef TILE_IDE_NO_WAIT
	u32 target_3;
#endif

	tile_ide_outl_program = gpio_program_address;

	/* Load PARAM_0 into CF_VM_REG_0. */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_LW,
				       CF_VM_REG_0,
				       CF_VM_CONST_PARAM_0,
				       CF_VM_CONST_0);

	/* Set the control bits to their starting point (passed in). */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_SET_PINS,
				       cf_ctl2_bank,
				       CF_VM_REG_0,
				       CF_VM_CONST_CTL2_MASK);

	/* Load the data into CF_VM_REG_0. */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_LW,
				       CF_VM_REG_0,
				       CF_VM_CONST_PARAM_1,
				       CF_VM_CONST_0);

	/* Put the data on the bus. */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_SET_PINS,
				       cf_data_bank,
				       CF_VM_REG_0,
				       CF_VM_CONST_FFFFFFFF);

	/* Assert IOWR_L (set it to 0). */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_SET_PINS,
				       cf_ctl2_bank,
				       CF_VM_CONST_0,
				       CF_VM_CONST_IOWR_L_BIT);
#ifndef TILE_IDE_NO_WAIT
	/* read the wait bit/bank */
	target_3 = gpio_program_address;
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address, GPIO_READ_PINS,
				       CF_VM_REG_0, cf_ctl2_bank, 0);
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address, GPIO_AND,
				       CF_VM_REG_0, CF_VM_REG_0,
				       CF_VM_CONST_IORDY_BIT);
	gpio_program_address =
	    tile_ide_write_immediate_instruction(gpio_program_address, GPIO_BZ,
						 CF_VM_REG_0,
						 (target_3 -
						  gpio_program_address) /
						 sizeof(gpio_instruction_t));
#endif

	/* Deassert IOWR_L (set it to 1). */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_SET_PINS,
				       cf_ctl2_bank,
				       CF_VM_CONST_IOWR_L_BIT,
				       CF_VM_CONST_IOWR_L_BIT);

	/* Put the other signal pins back to their resting values. */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_SET_PINS,
				       cf_ctl2_bank,
				       CF_VM_CONST_DONE,
				       CF_VM_CONST_CTL2_MASK);

	/* And we're done. */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_DONE,
				       CF_VM_REG_0,
				       CF_VM_REG_0,
				       CF_VM_REG_0);
	return gpio_program_address;
}

static u32 tile_ide_outsw_program;

static u32 tile_ide_compose_outsw_program(u32 gpio_program_address)
{
	u32 target_2;
#ifndef TILE_IDE_NO_WAIT
	u32 target_3;
#endif
	tile_ide_outsw_program = gpio_program_address;

	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_LW,
				       CF_VM_REG_3,
				       CF_VM_CONST_PARAM_0,
				       CF_VM_CONST_0);

	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_SET_PINS,
				       cf_ctl2_bank,
				       CF_VM_REG_3,
				       CF_VM_CONST_CTL2_MASK);

	/* CF_VM_REG_1 contains the write length. */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_LW,
				       CF_VM_REG_1,
				       CF_VM_CONST_PARAM_1,
				       CF_VM_CONST_0);

	/* CF_VM_REG_2 contains the address */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_ADD,
				       CF_VM_REG_2,
				       CF_VM_CONST_0,
				       CF_VM_CONST_0);

	/*
	 * There are already three parameters in the write space, so
	 *  use start + 12.
	 */
	gpio_program_address =
	    tile_ide_write_immediate_instruction(gpio_program_address,
						 GPIO_LI,
						 CF_VM_REG_2,
						 GPIO_CTL_WRITE_OFF + 12);

	/* TARGET 2 */
	/* Load the data into CF_VM_REG_0. */
	target_2 = gpio_program_address;
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_LH,
				       CF_VM_REG_0,
				       CF_VM_REG_2,
				       CF_VM_CONST_0);

	/* Put the data on the bus. */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_SET_PINS,
				       cf_data_bank,
				       CF_VM_REG_0,
				       CF_VM_CONST_FFFFFFFF);

	/* Asset IOWR_L (set it to 0). */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_SET_PINS,
				       cf_ctl2_bank,
				       CF_VM_CONST_0,
				       CF_VM_CONST_IOWR_L_BIT);

#ifndef TILE_IDE_NO_WAIT
	/* read the wait bit/bank */
	target_3 = gpio_program_address;
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_READ_PINS,
				       CF_VM_REG_0,
				       cf_ctl2_bank,
				       0);
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_AND,
				       CF_VM_REG_0,
				       CF_VM_REG_0,
				       CF_VM_CONST_IORDY_BIT);
	gpio_program_address =
	    tile_ide_write_immediate_instruction(gpio_program_address,
						 GPIO_BZ,
						 CF_VM_REG_0,
						 (target_3 -
						  gpio_program_address) /
						 sizeof(gpio_instruction_t));
#endif

	/* Deassert IOOWR_L (set it to 1). */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_SET_PINS,
				       cf_ctl2_bank,
				       CF_VM_CONST_IOWR_L_BIT,
				       CF_VM_CONST_IOWR_L_BIT);

	/* Decrement the byte counter. */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_ADD,
				       CF_VM_REG_1,
				       CF_VM_REG_1,
				       CF_VM_CONST_FFFFFFFF);

	/* Increment the address. */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_ADD,
				       CF_VM_REG_2,
				       CF_VM_REG_2,
				       CF_VM_CONST_2);

	/* Branch to target_2 if we have hit zero. */
	gpio_program_address =
	    tile_ide_write_immediate_instruction(gpio_program_address,
						 GPIO_BNZ, CF_VM_REG_1,
						 (target_2 -
						  gpio_program_address) /
						 sizeof(gpio_instruction_t));

	/* Put data bus into read mode. */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_SET_PINS,
				       cf_ctl2_bank,
				       CF_VM_CONST_DONE,
				       CF_VM_CONST_CTL2_MASK);

	/* And we're done. */
	gpio_program_address =
	    tile_ide_write_instruction(gpio_program_address,
				       GPIO_DONE,
				       CF_VM_REG_0,
				       CF_VM_REG_0,
				       CF_VM_REG_0);

	return gpio_program_address;
}

static void tile_ide_outsw(unsigned long port, void *addr, u32 count)
{
	const u16 *p = addr;
	u32 transaction_size;
	u32 buffer[3];

	buffer[0] = tile_ide_outsw_program;
	buffer[1] =
		((port & 0x7) << CF_ADDR_BIT) |
		CLR_BIT(CF_CS0_L_BIT)	      |
		SET_BIT(CF_IORD_L_BIT)	      |
		SET_BIT(CF_IOWR_L_BIT);
	TIDE_DEBUG(DRV_NAME ": outsw port: 0x%x addr: %x count: %d\n",
		   port, addr, count);
	while (count) {
		transaction_size = (count > (TILE_IDE_MAX_TRANSFER_BYTE_SIZE /
					     sizeof(u16)))
		    ? (TILE_IDE_MAX_TRANSFER_BYTE_SIZE / sizeof(u16)) : count;
		buffer[2] = transaction_size;
		if (hv_dev_pwrite(tide_satchel.dev_fds, 0, (HV_VirtAddr)p,
				  (transaction_size *
				   sizeof(u16)), GPIO_CTL_WRITE_OFF + 12) !=
		    (transaction_size * sizeof(u16))) {
			pr_err(DRV_NAME ":HV GPIO problem: outsw "
			       "write program problem\n");
		}
		if (hv_dev_pwrite(tide_satchel.dev_fds, 0,
				  (HV_VirtAddr)&buffer[0], 12,
				  GPIO_CTL_WRITE_OFF) != 12) {
			pr_err(DRV_NAME ":HV GPIO problem: outsw "
			       "write program problem\n");
		}
		p += transaction_size;
		count -= transaction_size;
	}
}

static void tile_ide_config_constant(u32 offset, u32 buffer)
{
	hv_dev_pwrite(tide_satchel.dev_fds, 0, (HV_VirtAddr)&buffer,
		      4, GPIO_CTL_CONFIG_RF_OFF + (sizeof(u32) * offset));
}

#else /* TIDE_USE_HYPERVISOR_PROGRAMS */

static u32 tile_ide_inl(unsigned long port)
{
	u32 buffer;

	/* Put data bus into read mode. */
	tile_ide_release_bits(0xffffFFFF, cf_data_bank);

	tile_ide_set_bits(((port & 0x7) << CF_ADDR_BIT) |
			  cf_ctl2_rw_setup_bits,
			  cf_ctl2_mask,
			  cf_ctl2_bank);

	/* Assert IORD_L to trigger the read. */
	tile_ide_set_bits(CLR_BIT(CF_IORD_L_BIT),
			  MASK_BIT(CF_IORD_L_BIT),
			  cf_ctl2_bank);

#ifndef TILE_IDE_NO_WAIT
	/* read the wait signal */
	while (!(tile_ide_read_bits(cf_ctl2_bank) & (1 << CF_IORDY_BIT)))
		;
#endif

	/* Read the data. */
	buffer = tile_ide_read_bits(cf_data_bank);

	tile_ide_set_bits(cf_ctl2_default_bits,
			  cf_ctl2_mask,
			  cf_ctl2_bank);

	TIDE_DEBUG(DRV_NAME ": inl port: 0x%x return: 0x%x\n", port, buffer);

	return (u32) buffer;
}

static void tile_ide_insw(unsigned long port, void *addr, u32 count)
{
	u16 *p = addr;
	TIDE_DEBUG(DRV_NAME ": insw port: 0x%x addr: 0x%x count %d\n",
		   port, addr, count);
	while (count--)
		*p++ = tile_ide_inl(port);
}

static void tile_ide_outl(u32 val, unsigned long port)
{
	TIDE_DEBUG(DRV_NAME ": outl val: %d port: 0x%x\n", val, port);

	/* We want CS1 high and CS0 low. */
	tile_ide_set_bits(((port & 0x7) << CF_ADDR_BIT) |
			  cf_ctl2_rw_setup_bits,
			  cf_ctl2_mask,
			  cf_ctl2_bank);

	/* Put the data on the bus */
	tile_ide_set_bits(val, 0xFFFFffff, cf_data_bank);

	/* Assert IOWR_L to trigger the write. */
	tile_ide_set_bits(CLR_BIT(CF_IOWR_L_BIT),
			  MASK_BIT(CF_IOWR_L_BIT),
			  cf_ctl2_bank);

#ifndef TILE_IDE_NO_WAIT
	/* read the wait signal */
	while (!(tile_ide_read_bits(cf_ctl2_bank) & (1 << CF_IORDY_BIT)))
		;
#endif

	/* Deassert IOWR_L. */
	tile_ide_set_bits(SET_BIT(CF_IOWR_L_BIT),
			  MASK_BIT(CF_IOWR_L_BIT),
			  cf_ctl2_bank);
	/* Return the rest of the control bits to their default values. */
	tile_ide_set_bits(cf_ctl2_default_bits,
			  cf_ctl2_mask,
			  cf_ctl2_bank);

	/* put data bus into read mode */
	tile_ide_release_bits(0xffffFFFF, cf_data_bank);
}

static void tile_ide_outsw(unsigned long port, void *addr, u32 count)
{
	const unsigned short *p = addr;
	TIDE_DEBUG(DRV_NAME ": outsw port: 0x%x addr: %x count: %d\n",
		   port, addr, count);
	while (count--)
		tile_ide_outl(*p++, port);
}

#endif /* TIDE_USE_HYPERVISOR_PROGRAMS */

static u8 tile_ide_inb(unsigned long port)
{
	return tile_ide_inl(port);
}

static void tile_ide_outb(u8 val, unsigned long port)
{
	tile_ide_outl(val, port);
}

/*
 * Verify that we have access to the bits we need.  If we don't then
 * there's no point in attempting to start the driver.
 */
static int test_gpio_access(void)
{
	u32 buffer;

	/* Read a bank of pins. */
	buffer = tile_ide_read_bits(cf_ctl1_bank) & CF_CTL1_MASK;
	/* Attempt to write all the bits we need with their current values. */
	if (tile_ide_set_bits(buffer, CF_CTL1_MASK, cf_ctl1_bank)) {
		TIDE_DEBUG(DRV_NAME ": GPIO failure on port %d, no access"
			   "to all of these bits: 0x%08X\n", cf_ctl1_bank,
			   CF_CTL1_MASK);
		return -1;
	}
	/* Now do the same on the next bank of pins. */
	buffer = tile_ide_read_bits(cf_ctl2_bank) & cf_ctl2_mask;
	if (tile_ide_set_bits(buffer, cf_ctl2_mask, cf_ctl2_bank)) {
		TIDE_DEBUG(DRV_NAME ": GPIO failure on port %d, no access"
			   "to all of these bits: 0x%08X\n", cf_ctl2_bank,
			   cf_ctl2_mask);
		return -1;
	}
	/* And the same for the data pins. */
	buffer = tile_ide_read_bits(cf_data_bank);
	if (tile_ide_set_bits(buffer, 0xFFFF, cf_data_bank)) {
		TIDE_DEBUG(DRV_NAME ": GPIO failure on port %d, no access"
			   "to all of these bits: 0x%08X\n", cf_data_bank,
			   0xFFFF);
		return -1;
	}

	return 0;
}

/*
 * Check for the presence of an IDE device.  If there is no device
 * present then we do not need to continue with driver and IDE
 * subsystem initialization, which saves boot time.
 */
static int check_for_device_present(void)
{
	int reg0;
	int n;

	/*
	 * Read registers 0 through 7.	If the same value is seen in
	 * all registers then assume there is no device on the
	 * interface.
	 */
	reg0 = tile_ide_inl(0);
	for (n = 1; n < 8; n++)
		if (tile_ide_inl(n) != reg0)
			return 1;

	return 0;
}

/*
 * tile_ide_open_gpio_driver - turn on the gpio driver
 *
 * Returns the IRQ allocated for GPIO.
 */
static int tile_ide_open_gpio_driver(u32 port)
{
	u32 buffer[4];
	gpio_irq_pin_t irq_pin;
	gpio_irq_number_t irq_number;
	u32 counter;
	u32 little_program_location;
	char file[128];
	int device_present;

	/* try to open the GPIO driver */
	sprintf(file, "gpio/%d", port);
	tide_satchel.dev_fds = hv_dev_open((HV_VirtAddr)file, 0);
	if (tide_satchel.dev_fds < 0) {
		pr_info(DRV_NAME ": no such device %s\n", file);
		return -1;
	}

	/* Figure out which IRQ the GPIO pin interrupt will trigger. */
	hv_dev_pread(tide_satchel.dev_fds, 0, (HV_VirtAddr)&irq_number,
		     sizeof(irq_number), GPIO_CTL_IRQ_NUMBER_OFF);
	tile_irq_activate(irq_number.irq_number, TILE_IRQ_SW_CLEAR);

	if (hv_dev_pread(tide_satchel.dev_fds, 0, (HV_VirtAddr)&buffer,
			 sizeof(u32), GPIO_CTL_BOARD_WIRING_OFF) !=
	    sizeof(u32)) {
		pr_info(DRV_NAME ": error reading board wiring "
		       "code\n");
		return -1;
	} else {
		switch (buffer[0]) {
		case GPIO_BOARD_WIRING_IDE_REV1:
			pr_info(DRV_NAME ": GPIO wiring rev1 not "
			       "supported\n");
			return -1;

		case GPIO_BOARD_WIRING_IDE_REV2:
			pr_info(DRV_NAME ": GPIO wiring rev2 not "
			       "supported\n");
			return -1;

		case GPIO_BOARD_WIRING_IDE_TE20G_V2:
			cf_ctl1_bank = 0;
			cf_ctl2_bank = 3;
			cf_data_bank = 1;
			break;

		case GPIO_BOARD_WIRING_IDE_AMC_V1:
			cf_ctl1_bank = 0;
			cf_ctl2_bank = 2;
			cf_data_bank = 3;
			break;

		case GPIO_BOARD_WIRING_IDE_BLADE_V1:
			cf_ctl1_bank = 2;
			cf_ctl2_bank = 3;
			cf_data_bank = 1;
			break;

		case GPIO_BOARD_WIRING_IDE_BLADE_V2:
			cf_ctl1_bank = 2;
			cf_ctl2_bank = 3;
			cf_data_bank = 1;
			cf_has_dma_bits = 0;
			cf_ctl2_mask = CF_CTL2_MASK_NODMA;
			cf_ctl2_const_bits = CF_CTL2_CONST_BITS_NODMA;
			cf_ctl2_rw_setup_bits = CF_CTL2_RW_SETUP_BITS_NODMA;
			cf_ctl2_default_bits = CF_CTL2_DEFAULT_BITS_NODMA;
			break;

		default:
		case GPIO_BOARD_WIRING_UNKNOWN:
			pr_info(DRV_NAME ": Unsupported GPIO "
			       "configuration\n");
			return -1;
		}
	}

	if (test_gpio_access()) {
		pr_info(DRV_NAME
		       ": Unsupported GPIO configuration.\n");
		return -1;
	}

	/* Set up our IRQ pin. */
	irq_pin.pin_bank = cf_ctl1_bank;
	irq_pin.pin_location = CF_INTRQ_BIT;
	hv_dev_pwrite(tide_satchel.dev_fds, 0, (HV_VirtAddr)&irq_pin,
		      sizeof(irq_pin), GPIO_CTL_IRQ_PIN_OFF);

	/* load up the GPIO controller with interesting constants */
	for (counter = 0; counter < GPIO_NUMBER_OF_REGISTERS; counter++) {
		hv_dev_pwrite(tide_satchel.dev_fds, 0,
			      (HV_VirtAddr)&counter, 4,
			      GPIO_CTL_CONFIG_RF_OFF +
			      (sizeof(u32) * counter));
	}

	/* Pull reset LOW to hold the chip in reset. */
	tile_ide_set_bits(CLR_BIT(CF_RESET_BIT),
			  MASK_BIT(CF_RESET_BIT),
			  cf_ctl1_bank);

#ifdef TIDE_USE_HYPERVISOR_PROGRAMS
	/* Set up constants */
	tile_ide_config_constant(CF_VM_CONST_FFFFFFFF, 0xffffFFFF);

	tile_ide_config_constant(CF_VM_CONST_PARAM_0, GPIO_CTL_WRITE_OFF + 4);
	tile_ide_config_constant(CF_VM_CONST_PARAM_1, GPIO_CTL_WRITE_OFF + 8);
	tile_ide_config_constant(CF_VM_CONST_PARAM_2, GPIO_CTL_WRITE_OFF + 12);
	tile_ide_config_constant(CF_VM_CONST_PARAM_3, GPIO_CTL_WRITE_OFF + 16);
	tile_ide_config_constant(CF_VM_CONST_PARAM_4, GPIO_CTL_WRITE_OFF + 20);
	tile_ide_config_constant(CF_VM_CONST_IORD_L_BIT,
				 SET_BIT(CF_IORD_L_BIT));
	tile_ide_config_constant(CF_VM_CONST_IORDY_BIT,
				 SET_BIT(CF_IORDY_BIT));
	tile_ide_config_constant(CF_VM_CONST_IOWR_L_BIT,
				 SET_BIT(CF_IOWR_L_BIT));
	tile_ide_config_constant(CF_VM_CONST_CTL2_MASK,
				 CF_ADDR_MASK		 |
				 MASK_BIT(CF_IORD_L_BIT) |
				 MASK_BIT(CF_IOWR_L_BIT) |
				 MASK_BIT(CF_CS0_L_BIT));
	tile_ide_config_constant(CF_VM_CONST_DONE,
				0                      | /* Address = 0 */
				SET_BIT(CF_IORD_L_BIT) |
				SET_BIT(CF_IOWR_L_BIT) |
				SET_BIT(CF_CS1_L_BIT)  |
				SET_BIT(CF_CS0_L_BIT));

	/* Set up little programs */
	little_program_location = GPIO_CTL_PROGRAM_STORE_OFF;
	little_program_location =
	    tile_ide_compose_inl_program(little_program_location);
	little_program_location =
	    tile_ide_compose_outl_program(little_program_location);
	little_program_location =
	    tile_ide_compose_insw_program(little_program_location);
	little_program_location =
	    tile_ide_compose_outsw_program(little_program_location);

#endif /* TIDE_USE_HYPERVISOR_PROGRAM */


	/*
	 * Set up the starting value for the various control bits.
	 * CF_CTL2_READ_DONE_BITS is a good starting point.
	 */
	tile_ide_set_bits(cf_ctl2_default_bits,
			  cf_ctl2_mask,
			  cf_ctl2_bank);

	/* Release IORDY */
	tile_ide_release_bits(MASK_BIT(CF_IORDY_BIT), cf_ctl2_bank);

	/* Release CF_INTRQ */
	tile_ide_release_bits(MASK_BIT(CF_INTRQ_BIT), cf_ctl1_bank);

	/* Release D0-15 */
	tile_ide_release_bits(0xffffFFFF, cf_data_bank);

	/* Release DMARQ */
	if (cf_has_dma_bits)
		tile_ide_release_bits(MASK_BIT(CF_DMARQ_BIT), cf_ctl2_bank);

	/* Release IOCS16 */
	tile_ide_release_bits(MASK_BIT(CF_IOCS16_L_BIT), cf_ctl2_bank);

	/* Release DASP# */
	tile_ide_release_bits(MASK_BIT(CF_DASP_L_BIT), cf_ctl1_bank);

	/* Release PDIAG# */
	tile_ide_release_bits(MASK_BIT(CF_PDIAG_L_BIT), cf_ctl1_bank);

	/* put reset HIGH to take the chip out of reset. */
	tile_ide_set_bits(SET_BIT(CF_RESET_BIT),
			  MASK_BIT(CF_RESET_BIT),
			  cf_ctl1_bank);

	/* Wait up to a second to see if the device is responding. */
	for (counter = 0; counter < 100; counter++) {
		device_present = check_for_device_present();
		if (device_present)
			break;
		mdelay(10);
	}

	if (!device_present) {
		/*
		 * If we're not going to initialize the driver then
		 * release all the pins.
		 */
		tile_ide_release_bits(CF_CTL1_MASK, cf_ctl1_bank);
		tile_ide_release_bits(cf_ctl2_mask, cf_ctl2_bank);
		tile_ide_release_bits(0xFFFF, cf_data_bank);

		pr_info(DRV_NAME ": No IDE device.\n");

		return -1;
	}

	return irq_number.irq_number;
}


/*
 * The following routines tile_XXX() are all just copies of ide_XXX()
 * without the "mmio" and "32bit" support, and directly calling the
 * tile IDE routines for input and output of data.  The tile_tp_ops
 * structure contains all the pointers to these routines.
 */

static void tile_exec_command(ide_hwif_t *hwif, u8 cmd)
{
	tile_ide_outb(cmd, hwif->io_ports.command_addr);
}

static u8 tile_read_status(ide_hwif_t *hwif)
{
	return tile_ide_inb(hwif->io_ports.status_addr);
}

static u8 tile_read_altstatus(ide_hwif_t *hwif)
{
	return tile_ide_inb(hwif->io_ports.ctl_addr);
}

static void tile_write_devctl(ide_hwif_t *hwif, u8 ctl)
{
	tile_ide_outb(ctl, hwif->io_ports.ctl_addr);
}

static void tile_dev_select(ide_drive_t *drive)
{
	ide_hwif_t *hwif = drive->hwif;
	u8 select = drive->select | ATA_DEVICE_OBS;

	tile_ide_outb(select, hwif->io_ports.device_addr);
}

static void tile_tf_load(ide_drive_t *drive, struct ide_taskfile *tf, u8 valid)
{
	ide_hwif_t *hwif = drive->hwif;
	struct ide_io_ports *io_ports = &hwif->io_ports;

	if (valid & IDE_VALID_FEATURE)
		tile_ide_outb(tf->feature, io_ports->feature_addr);
	if (valid & IDE_VALID_NSECT)
		tile_ide_outb(tf->nsect, io_ports->nsect_addr);
	if (valid & IDE_VALID_LBAL)
		tile_ide_outb(tf->lbal, io_ports->lbal_addr);
	if (valid & IDE_VALID_LBAM)
		tile_ide_outb(tf->lbam, io_ports->lbam_addr);
	if (valid & IDE_VALID_LBAH)
		tile_ide_outb(tf->lbah, io_ports->lbah_addr);
	if (valid & IDE_VALID_DEVICE)
		tile_ide_outb(tf->device, io_ports->device_addr);
}

static void tile_tf_read(ide_drive_t *drive, struct ide_taskfile *tf, u8 valid)
{
	ide_hwif_t *hwif = drive->hwif;
	struct ide_io_ports *io_ports = &hwif->io_ports;

	if (valid & IDE_VALID_ERROR)
		tf->error  = tile_ide_inb(io_ports->feature_addr);
	if (valid & IDE_VALID_NSECT)
		tf->nsect  = tile_ide_inb(io_ports->nsect_addr);
	if (valid & IDE_VALID_LBAL)
		tf->lbal   = tile_ide_inb(io_ports->lbal_addr);
	if (valid & IDE_VALID_LBAM)
		tf->lbam   = tile_ide_inb(io_ports->lbam_addr);
	if (valid & IDE_VALID_LBAH)
		tf->lbah   = tile_ide_inb(io_ports->lbah_addr);
	if (valid & IDE_VALID_DEVICE)
		tf->device = tile_ide_inb(io_ports->device_addr);
}

static void tile_input_data(ide_drive_t *drive, struct ide_cmd *cmd,
			    void *buf, unsigned int len)
{
	ide_hwif_t *hwif = drive->hwif;
	struct ide_io_ports *io_ports = &hwif->io_ports;
	unsigned long data_addr = io_ports->data_addr;
	unsigned int words = (len + 1) >> 1;

	tile_ide_insw(data_addr, buf, words);
}

static void tile_output_data(ide_drive_t *drive, struct ide_cmd *cmd,
			     void *buf, unsigned int len)
{
	ide_hwif_t *hwif = drive->hwif;
	struct ide_io_ports *io_ports = &hwif->io_ports;
	unsigned long data_addr = io_ports->data_addr;
	unsigned int words = (len + 1) >> 1;

	tile_ide_outsw(data_addr, buf, words);
}

static const struct ide_tp_ops tile_tp_ops = {
	.exec_command		= tile_exec_command,
	.read_status		= tile_read_status,
	.read_altstatus		= tile_read_altstatus,
	.write_devctl		= tile_write_devctl,

	.dev_select		= tile_dev_select,
	.tf_load		= tile_tf_load,
	.tf_read		= tile_tf_read,

	.input_data		= tile_input_data,
	.output_data		= tile_output_data,
};

static void tile_hw_setup(struct ide_hw *hw, int irq)
{
	memset(hw, 0, sizeof(*hw));
	ide_std_init_ports(hw, 0x100, 0xe + 0x100);
	hw->irq = irq;
	hw->io_ports.ctl_addr = 0;
}

static const struct ide_port_info tile_port_info = {
	.name		= DRV_NAME,
	.chipset	= ide_generic,
	.tp_ops		= &tile_tp_ops,
	.host_flags	=
		IDE_HFLAG_NO_IO_32BIT |	/* No 32 bit IO */
		IDE_HFLAG_NO_DMA      |	/* DMA is not supported. */
		IDE_HFLAG_NO_ATAPI_DMA	/* DMA is really not supported. */
};

static int __init tile_ide_init(void)
{
	struct ide_host *host;
	struct ide_hw hw;
	struct ide_hw *hws[] = { &hw };
	int irq;
	int rc;

	/* Note that right now this only supports a single ide channel */
	irq = tile_ide_open_gpio_driver(0);
	if (irq == -1)
		return -ENOENT;

	tile_hw_setup(&hw, irq);

	host = ide_host_alloc(&tile_port_info, hws, 1);
	if (host == NULL) {
		pr_err(DRV_NAME ": Couldn't register interface.\n");
		return -ENOMEM;
	}

	pr_info(DRV_NAME ": Tile IDE interface ide%d\n",
	       host->ports[0]->index);

	rc = ide_host_register(host, &tile_port_info, hws);
	if (rc)
		ide_host_free(host);
	return rc;
}

module_init(tile_ide_init);

MODULE_LICENSE("GPL");
