/*
 * Copyright 2011 Tilera Corporation. All Rights Reserved.
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation, version 2.
 *
 *   This program is distributed in the hope that it will be useful, but
 *   WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, GOOD TITLE or
 *   NON INFRINGEMENT.  See the GNU General Public License for
 *   more details.
 *
 * SPI Flash ROM driver
 *
 * This source code is derived from code provided in "Linux Device
 * Drivers" by Alessandro Rubini and Jonathan Corbet, published by
 * O'Reilly & Associates.
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>
#include <linux/kernel.h>	/* printk() */
#include <linux/slab.h>		/* kmalloc() */
#include <linux/fs.h>		/* everything... */
#include <linux/errno.h>	/* error codes */
#include <linux/types.h>	/* size_t */
#include <linux/proc_fs.h>
#include <linux/fcntl.h>	/* O_ACCMODE */
#include <linux/aio.h>
#include <linux/pagemap.h>
#include <linux/hugetlb.h>
#include <linux/uaccess.h>
#include <hv/hypervisor.h>
#include <linux/ioctl.h>
#include <linux/cdev.h>
#include <linux/delay.h>
#include <hv/drv_srom_intf.h>

/*
 * Size of our hypervisor I/O requests.  We break up large transfers
 * so that we don't spend large uninterrupted spans of time in the
 * hypervisor.  Erasing an SROM sector takes a significant fraction of
 * a second, so if we allowed the user to, say, do one I/O to write the
 * entire ROM, we'd get soft lockup timeouts, or worse.
 */
#define SROM_CHUNK_SIZE 4096

/*
 * When hypervisor is busy (e.g. erasing), poll the status periodically.
 */

/*
 * Interval to poll the state in msec
 */
#define SROM_WAIT_TRY_INTERVAL 20

/*
 * Maximum times to poll the state
 */
#define SROM_MAX_WAIT_TRY_TIMES 1000

struct srom_dev {
	struct cdev cdev;		/* Character device structure */
	int hv_devhdl;			/* Handle for hypervisor device */
	u32 size;		        /* Size of this device */
	char *info_string;		/* String returned by the info device
					   if this is it; NULL otherwise */
	struct mutex srom_lock;		/* Allow only one accessor at a time */
};

static int srom_major;		/* Dynamic major by default */
static int srom_devs =    4;	/* One per SROM partition plus one info file */

/* Minor number of the info file */
static inline int srom_info_minor(void)
{
	return srom_devs - 1;
}

module_param(srom_major, int, 0);
module_param(srom_devs, int, 0);
MODULE_AUTHOR("Tilera Corporation");
MODULE_LICENSE("Dual BSD/GPL");

static struct srom_dev *srom_devices; /* allocated in srom_init */

/**
 * Macro to complete a read/write transaction. Returns last hv
 * syscall return value.
 */
#define SROM_RD_OR_WR(func, ...) \
	({								\
	int _hv_retval = func(__VA_ARGS__);				\
	int retries;							\
									\
	while (_hv_retval < 0) {					\
		if (_hv_retval == HV_EBUSY) {				\
			retries = SROM_MAX_WAIT_TRY_TIMES;		\
									\
			while (retries > 0) {				\
				_hv_retval = func(__VA_ARGS__);		\
				if (_hv_retval == HV_EBUSY) {		\
					msleep(SROM_WAIT_TRY_INTERVAL);	\
					retries--;			\
				} else {				\
					break;				\
				}					\
			}						\
									\
			if (_hv_retval == HV_EBUSY) {			\
				pr_err("srom: "#func" failed ("		\
				       "timeout)\n");			\
				retval = -EIO;				\
				goto op_done;				\
			}						\
		} else if (_hv_retval == HV_EAGAIN) {			\
			_hv_retval = func(__VA_ARGS__);			\
		} else {						\
			pr_err("srom: "#func" failed, "			\
				"error %d\n", _hv_retval);		\
			retval = -EIO;					\
			goto op_done;					\
		}							\
	}								\
	_hv_retval;							\
	})

/**
 * srom_open() - Device open routine.
 * @inode: Inode for this device.
 * @filp: File for this specific open of the device.
 *
 * Returns zero, or an error code.
 */
static int srom_open(struct inode *inode, struct file *filp)
{
	struct srom_dev *dev;

	/* Find the device */
	dev = container_of(inode->i_cdev, struct srom_dev, cdev);

	/* Now open the hypervisor device if we haven't already. */
	if (dev->hv_devhdl == 0) {
		char buf[20];
		int instance = iminor(inode);
		if (instance != srom_info_minor()) {
			sprintf(buf, "srom/0/%d", instance);
			dev->hv_devhdl = hv_dev_open((HV_VirtAddr)buf, 0);
			if (dev->hv_devhdl > 0) {
				dev->size = 0;
				if (mutex_lock_interruptible(&dev->srom_lock))
					return -ERESTARTSYS;
				hv_dev_pread(dev->hv_devhdl, 0,
					     (HV_VirtAddr)&dev->size,
					     sizeof(dev->size),
					     SROM_TOTAL_SIZE_OFF);
				mutex_unlock(&dev->srom_lock);
			}
		} else {
			u32 sector_size = 0;
			u32 page_size = 0;

			sprintf(buf, "srom/0/0");
			dev->hv_devhdl = hv_dev_open((HV_VirtAddr)buf, 0);
			if (dev->hv_devhdl > 0) {
				static const int info_string_size = 80;

				if (mutex_lock_interruptible(&dev->srom_lock))
					return -ERESTARTSYS;
				hv_dev_pread(dev->hv_devhdl, 0,
					     (HV_VirtAddr)&sector_size,
					     sizeof(sector_size),
					     SROM_SECTOR_SIZE_OFF);
				hv_dev_pread(dev->hv_devhdl, 0,
					     (HV_VirtAddr)&page_size,
					     sizeof(page_size),
					     SROM_PAGE_SIZE_OFF);
				mutex_unlock(&dev->srom_lock);

				dev->info_string = kmalloc(info_string_size,
							GFP_KERNEL);
				snprintf(dev->info_string, info_string_size,
					"sector_size: %d\npage_size: %d\n",
					sector_size, page_size);
			}
		}
	}

	/* If we tried and failed to open it, fail. */
	if (dev->hv_devhdl < 0) {
		switch (dev->hv_devhdl)	{
		case HV_ENODEV:
			return -ENODEV;
		default:
			return (ssize_t)dev->hv_devhdl;
		}
	}

	filp->private_data = dev;

	return 0;          /* success */
}


/**
 * srom_release() - Device release routine.
 * @inode: Inode for this device.
 * @filp: File for this specific open of the device.
 *
 * Returns zero, or an error code.
 */
static int srom_release(struct inode *inode, struct file *filp)
{
	struct srom_dev *dev = filp->private_data;
	int retval = 0;
	char dummy;

	/*
	 * First we need to make sure we've flushed anything written to
	 * the ROM.
	 */
	if (mutex_lock_interruptible(&dev->srom_lock)) {
		retval = -ERESTARTSYS;
		filp->private_data = NULL;
		return retval;
	}

	if (dev->hv_devhdl > 0) {
		SROM_RD_OR_WR(hv_dev_pwrite, dev->hv_devhdl,
			      0, (HV_VirtAddr)&dummy, 1, SROM_FLUSH_OFF);
	}

op_done:
	mutex_unlock(&dev->srom_lock);
	filp->private_data = NULL;

	return retval;
}


/**
 * srom_read() - Read (control) data from the device.
 * @filp: File for this specific open of the device.
 * @buf: User's data buffer.
 * @count: Number of bytes requested.
 * @f_pos: File position.
 *
 * Returns number of bytes read, or an error code.
 */
static ssize_t srom_read(struct file *filp, char __user *buf,
			 size_t count, loff_t *f_pos)
{
	int retval = 0;
	void *kernbuf;
	struct srom_dev *dev = filp->private_data;

	if (dev->hv_devhdl < 0)
		return -EINVAL;

	if (dev->info_string) {
		int info_len = strlen(dev->info_string);
		int bytes_avail = info_len - *f_pos;
		int xfer_len = (bytes_avail < count) ? bytes_avail : count;

		if (xfer_len <= 0)
			return 0;

		if (copy_to_user(buf, dev->info_string + *f_pos, xfer_len))
			return -EFAULT;
		*f_pos += xfer_len;
		return xfer_len;
	}

	kernbuf = kmalloc(SROM_CHUNK_SIZE, GFP_KERNEL);
	if (!kernbuf)
		return -ENOMEM;

	if (mutex_lock_interruptible(&dev->srom_lock)) {
		retval = -ERESTARTSYS;
		kfree(kernbuf);
		return retval;
	}

	while (count) {
		int hv_retval;
		int bytes_this_pass = count;

		if (bytes_this_pass > SROM_CHUNK_SIZE)
			bytes_this_pass = SROM_CHUNK_SIZE;

		hv_retval = SROM_RD_OR_WR(hv_dev_pread, dev->hv_devhdl,
			      0, (HV_VirtAddr) kernbuf, bytes_this_pass,
			      *f_pos);

		if (hv_retval > 0) {
			if (copy_to_user(buf, kernbuf, hv_retval) != 0) {
				retval = -EFAULT;
				break;
			}
		} else if (hv_retval == 0) {
			break;
		}

		retval += hv_retval;
		*f_pos += hv_retval;
		buf += hv_retval;
		count -= hv_retval;
	}

op_done:
	mutex_unlock(&dev->srom_lock);
	kfree(kernbuf);

	return retval;
}

/**
 * srom_write() - Write (control) data to the device.
 * @filp: File for this specific open of the device.
 * @buf: User's data buffer.
 * @count: Number of bytes requested.
 * @f_pos: File position.
 *
 * Returns number of bytes written, or an error code.
 */
static ssize_t srom_write(struct file *filp, const char __user *buf,
			  size_t count, loff_t *f_pos)
{
	int retval = 0;
	void *kernbuf;
	struct srom_dev *dev = filp->private_data;

	if (dev->hv_devhdl < 0 || dev->info_string)
		return -EINVAL;

	kernbuf = kmalloc(SROM_CHUNK_SIZE, GFP_KERNEL);
	if (!kernbuf)
		return -ENOMEM;

	if (mutex_lock_interruptible(&dev->srom_lock)) {
		retval = -ERESTARTSYS;
		kfree(kernbuf);
		return retval;
	}

	while (count) {
		int hv_retval;
		int bytes_this_pass = count;

		if (bytes_this_pass > SROM_CHUNK_SIZE)
			bytes_this_pass = SROM_CHUNK_SIZE;

		if (copy_from_user(kernbuf, buf, bytes_this_pass) != 0) {
			retval = -EFAULT;
			break;
		}

		hv_retval = SROM_RD_OR_WR(hv_dev_pwrite, dev->hv_devhdl,
			      0, (HV_VirtAddr) kernbuf, bytes_this_pass,
			      *f_pos);

		if (hv_retval == 0)
			break;

		retval += hv_retval;
		*f_pos += hv_retval;
		buf += hv_retval;
		count -= hv_retval;
	}

op_done:
	mutex_unlock(&dev->srom_lock);
	kfree(kernbuf);

	return retval;
}

/**
 * srom_llseek() - Change the current device offset.
 * @filp: File for this specific open of the device.
 * @off: New offset value.
 * @whence: Base for new offset value.
 *
 * Returns new offset, or an error code.
 */
static loff_t srom_llseek(struct file *filp, loff_t off, int whence)
{
	struct srom_dev *dev = filp->private_data;
	long newpos;

	switch (whence) {
	case 0: /* SEEK_SET */
		newpos = off;
		break;

	case 1: /* SEEK_CUR */
		newpos = filp->f_pos + off;
		break;

	case 2: /* SEEK_END */
		newpos = dev->size + off;
		break;

	default: /* can't happen */
		return -EINVAL;
	}

	if (newpos < 0 || newpos > dev->size)
		return -EINVAL;

	filp->f_pos = newpos;
	return newpos;
}


/*
 * The fops
 */
static const struct file_operations srom_fops = {
	.owner =     THIS_MODULE,
	.llseek =    srom_llseek,
	.read =	     srom_read,
	.write =     srom_write,
	.open =	     srom_open,
	.release =   srom_release,
};

/**
 * srom_setup_cdev() - Set up a device instance in the cdev table.
 * @dev: Per-device SROM state.
 * @index: Device to set up.
 */
static void srom_setup_cdev(struct srom_dev *dev, int index)
{
	int err, devno = MKDEV(srom_major, index);

	cdev_init(&dev->cdev, &srom_fops);
	dev->cdev.owner = THIS_MODULE;
	dev->cdev.ops = &srom_fops;
	err = cdev_add(&dev->cdev, devno, 1);
	/* Fail gracefully if need be */
	if (err)
		pr_notice("Error %d adding srom%d", err, index);
}

/** srom_init() - Initialize the driver's module. */
static int srom_init(void)
{
	int result, i;
	dev_t dev = MKDEV(srom_major, 0);

	/*
	 * Register our major, and accept a dynamic number.
	 */
	if (srom_major)
		result = register_chrdev_region(dev, srom_devs, "srom");
	else {
		result = alloc_chrdev_region(&dev, 0, srom_devs, "srom");
		srom_major = MAJOR(dev);
	}
	if (result < 0)
		return result;


	/*
	 * Allocate the devices -- we can't have them static, as the number
	 * can be specified at load time.
	 */
	srom_devices = kzalloc(srom_devs * sizeof(struct srom_dev),
			       GFP_KERNEL);
	if (!srom_devices) {
		unregister_chrdev_region(dev, srom_devs);
		return -ENOMEM;
	}
	for (i = 0; i < srom_devs; i++) {
		srom_setup_cdev(srom_devices + i, i);
		mutex_init(&srom_devices[i].srom_lock);
	}

	return 0; /* succeed */
}

/** srom_cleanup() - Clean up the driver's module. */
static void srom_cleanup(void)
{
	int i;

	for (i = 0; i < srom_devs; i++) {
		cdev_del(&srom_devices[i].cdev);
		kfree(srom_devices[i].info_string);
	}
	kfree(srom_devices);
	unregister_chrdev_region(MKDEV(srom_major, 0), srom_devs);
}

module_init(srom_init);
module_exit(srom_cleanup);
