/*
 * tilepci_host_subset.c
 *
 * A minimal PCIe root complex driver for connecting to Tile
 * endpoints.  Provides support for boot, but not reset.  Also allows
 * reading or writing BAR-mapped memory, and determining the bus
 * address of a memory page on the RC side.
 *
 * Copyright 2011 Tilera Corporation. All Rights Reserved.
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation, version 2.
 *
 *   This program is distributed in the hope that it will be useful, but
 *   WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, GOOD TITLE or
 *   NON INFRINGEMENT.  See the GNU General Public License for
 *   more details.
 */
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/pci.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/cdev.h>
#include <linux/delay.h>
#include <linux/uaccess.h>
#include <linux/seq_file.h>
#include <linux/proc_fs.h>
#include <linux/mm.h>
#include <linux/pagemap.h>
#include <linux/hugetlb.h>
#include <linux/version.h>
#include <asm/pgtable.h>		/* io_remap_page_range */
#include <asm/tilepci.h>

#include "tilepci_host_subset.h"

static const char driver_name[] = DRIVER_NAME_STRING;

/**********************************************************************/
/*                          Boot Device Routines                      */
/**********************************************************************/
static ssize_t tlr_boot_write(struct file *filp, const char __user *buf,
			      size_t count, loff_t *f_pos);
static int tlr_boot_release(struct inode *inode, struct file *filp);

static const struct file_operations tlr_boot_ops = {
	.owner = THIS_MODULE,
	.write = tlr_boot_write,
	.release = tlr_boot_release
};



/*
 * Try to read the status register, which indicates whether a driver
 * is running.
 */
static int tlr_read_subsystem_flag(struct tlr_pcie_dev *tlr)
{
	u32 subsystem;

	pci_read_config_dword(tlr->pci_dev, PCI_SUBSYSTEM_VENDOR_ID,
			      &subsystem);
	return subsystem >> SUBSYSTEM_FLAG_SHIFT;
}

/*
 * We may need to wait a while for the device to boot.  While it is
 * booting, it will show up on PCI but not be able to process anything
 * beyond config transactions.  We don't enable the device or allow
 * open() calls to succeed until we get a successful read of the MMIO
 * status word.
 *
 * Return SUBSYSTEM_FLAG_IBOUND if the driver is ready,
 * SUBSYSTEM_FLAG_PENDING if the chip is booting but the driver isn't
 * quite up yet, and SUBSYSTEM_FLAG_RESET if there's no driver or
 * something went wrong.
 */
static u32 __tlr_check_ready(struct tlr_pcie_dev *tlr, int do_error_msg)
{
	unsigned long flags;
	u32 result = SUBSYSTEM_FLAG_RESET;

	/* First, do a fast-path check to see if the board is already
	 * up.  We're not going to transition anything yet, so we only
	 * need the spinlock. */
	spin_lock_irqsave(&tlr->is_ready_lock, flags);
	if (tlr->is_ready) {
		result = SUBSYSTEM_FLAG_IBOUND;
		goto finish;
	}

	if (!tlr->is_ready &&
	    !tlr->is_booting) {
		u32 value;

		value = tlr_read_subsystem_flag(tlr);
		if (value != SUBSYSTEM_FLAG_IBOUND) {
			/* Let caller know if we might be ready soon... */
			if (value == SUBSYSTEM_FLAG_PENDING)
				result = SUBSYSTEM_FLAG_PENDING;

			if (do_error_msg)
				ERR("TILE-side PCI driver did not respond; "
				    "is the driver running?\n");
			goto finish;
		}

		/* Enable PCI bus mastering; the TLR PCIE
		 * iBound gets upset if we query it without
		 * this being enabled. */
		pci_set_master(tlr->pci_dev);

		/* Read the version register, waking up the remote device. */
		readl(&tlr->regs->version);

		tlr->is_ready = 1;
	}

 finish:
	spin_unlock_irqrestore(&tlr->is_ready_lock, flags);
	return result;
}


/*
 * Call this version of __tlr_is_ready() whenever the user makes a
 * request that _requires_ the iBound to be up and running.
 */
static int tlr_is_ready(struct tlr_pcie_dev *tlr)
{
	return __tlr_check_ready(tlr, TRUE) == SUBSYSTEM_FLAG_IBOUND;
}

/*
 * Call this version of __tlr_is_ready() when the driver is just
 * poking the device on the off chance the Tile-side driver is up
 * (i.e. the user hasn't actually required any driver communication
 * paths yet).
 *
 * Unlike the other tlr_is_ready_ functions, this one returns true if
 * the Tile-side driver is still coming up and might be ready if we
 * retry later.
 */
static int tlr_is_pending(struct tlr_pcie_dev *tlr)
{
	return __tlr_check_ready(tlr, FALSE) == SUBSYSTEM_FLAG_PENDING;
}

/*
 * Don't allow booting if we're already booted once or the device
 * looks like a driver is already running.
 */
static int tlr_allow_boot(struct tlr_pcie_dev *tlr)
{
	return (__tlr_check_ready(tlr, FALSE) == SUBSYSTEM_FLAG_RESET) &&
		!tlr->booted;
}

/*
 * Reset the chip and get it ready to boot.  This function also
 * disables reading and writing of the stream devices until booting is
 * complete.
 */
static int tlr_boot_open(struct inode *inode, struct file *filp)
{
	struct tlr_pcie_dev *tlr =
		container_of(inode->i_cdev, struct tlr_pcie_dev, cdev);

	/* Set the private data to point at our state object. */
	filp->private_data = tlr;

	/* Use the stream read, write, etc. */
	filp->f_op = &tlr_boot_ops;

	/* One booter per device. */
	if (mutex_lock_interruptible(&tlr->boot_mutex))
		return -ERESTARTSYS;

	if (!tlr_allow_boot(tlr)) {
		mutex_unlock(&tlr->boot_mutex);
		return -ENODEV;
	}

	tlr->is_booting = 1;
	return 0;
}


/*
 * Each word written to the boot device is injected into BAR 0.
 */
static ssize_t tlr_boot_write(struct file *filp, const char __user *buf,
			      size_t count, loff_t *f_pos)
{
	struct tlr_pcie_dev *tlr = filp->private_data;
	size_t size = count & -4;
	size_t written;

	if (!access_ok(VERIFY_READ, buf, size))
		return -EFAULT;

	tlr->booted = 1;
	for (written = 0; written < size; written += sizeof(u32)) {
		u32 value;
		if (__get_user(value, (u32 __user *) (buf + written)))
			return -EFAULT;
		writel(value, &tlr->regs->status);
	}

	return written;
}

static int tlr_boot_release(struct inode *inode, struct file *filp)
{
	struct tlr_pcie_dev *tlr = filp->private_data;
	int sleep_msecs;
	const int POLLING_MSECS = 200;
	const int MAX_MSECS = 5000;

	/* FIXME: should we wait here or schedule a function to run later? */
	msleep(TLR_REBOOT_TO_READY_WAIT_MS);
	tlr->is_booting = 0;

	/* Poke the link to notify the endpoint driver that the BIOS
	 * state is ready.  If the poke function indicates that a
	 * driver is still initializing, wait up to MAX_MSECS for the
	 * driver to come up.  */
	sleep_msecs = 0;
	while (tlr_is_pending(tlr)) {
		if (sleep_msecs >= MAX_MSECS)
			break;
		msleep(POLLING_MSECS);
		sleep_msecs += POLLING_MSECS;
	}

	mutex_unlock(&tlr->boot_mutex);

	return 0;
}


/**********************************************************************/
/*                           BAR Memory Access                        */
/**********************************************************************/

#define BAR1_RESOURCE_INDEX 2

static resource_size_t tlr_barmem_size(struct tlr_pcie_dev *tlr)
{
	return pci_resource_len(tlr->pci_dev, BAR1_RESOURCE_INDEX);
}


static dma_addr_t tlr_barmem_base(struct tlr_pcie_dev *tlr)
{
	return pci_resource_start(tlr->pci_dev, BAR1_RESOURCE_INDEX);
}

/* Read 1, 2, or 4 bytes from a register. */
static ssize_t tlr_barmem_read(struct file *filp, char __user *buf,
		size_t count, loff_t *f_pos)
{
	struct tlr_pcie_dev *tlr = filp->private_data;
	loff_t pos = *f_pos;

	if (pos > tlr_barmem_size(tlr) - count + 1)
		return -EINVAL;

	if (!access_ok(VERIFY_WRITE, buf, count))
		return -EFAULT;

	switch (count) {
	case 1:
	{
		u8 value = readb(tlr->barmem + pos);
		if (__put_user(value, (u8 __user *)buf))
			return -EFAULT;
		break;
	}
	case 2:
	{
		u16 value = readw(tlr->barmem + pos);
		if (__put_user(value, (u16 __user *)buf))
			return -EFAULT;
		break;
	}
	case 4:
	{
		u32 value = readl(tlr->barmem + pos);
		if (__put_user(value, (u32 __user *)buf))
			return -EFAULT;
		break;
	}
	default:
		return -EINVAL;
	}

	return count;
}

/* Write 1, 2, or 4 bytes to a register. */
static ssize_t tlr_barmem_write(struct file *filp, const char __user *buf,
				size_t count, loff_t *f_pos)
{
	struct tlr_pcie_dev *tlr = filp->private_data;
	loff_t pos = *f_pos;

	if (pos > tlr_barmem_size(tlr) - count + 1)
		return -EINVAL;

	if (!access_ok(VERIFY_READ, buf, count))
		return -EFAULT;

	switch (count) {
	case 1:
	{
		u8 value;
		if (__get_user(value, (u8 __user *)buf))
			return -EFAULT;
		writeb(value, tlr->barmem + pos);
		break;
	}
	case 2:
	{
		u16 value;
		if (__get_user(value, (u16 __user *)buf))
			return -EFAULT;
		writew(value, tlr->barmem + pos);
		break;
	}
	case 4:
	{
		u32 value;
		if (__get_user(value, (u32 __user *)buf))
			return -EFAULT;
		writel(value, tlr->barmem + pos);
		break;
	}
	default:
		return -EINVAL;
	}

	return count;
}

/*
 * Helper routine for mapping single pages to the PCI bus.  All
 * registered pages are tracked on a "registered pages" list until
 * this file is closed.
 */
static int register_mem_buffer(struct tlr_pcie_dev *tlr, unsigned long va,
			       unsigned long size,
			       struct tlr_page_list_entry **entry_out)
{
	int count;
	struct vm_area_struct *vma;
	struct tlr_page_list_entry *entry;
	int page_size;
	unsigned long start_va;
	unsigned long next_page_va;
	int result;

	entry = kmalloc(sizeof(*entry), GFP_KERNEL);
	if (entry == NULL)
		return -ENOMEM;

	/* We only allow registration of a single page; getting
	 * multiple pages to be physically contiguous is difficult
	 * anyway. */
	down_read(&current->mm->mmap_sem);
	count = get_user_pages(current, current->mm, va, 1 /* num_pages */,
			       1 /* writable */, 0, &entry->page, &vma);
	up_read(&current->mm->mmap_sem);
	if (count != 1) {
		TRACE("get_user_pages() failure.\n");
		result = -EFAULT;
		goto get_user_pages_failed;
	}

	page_size = is_vm_hugetlb_page(vma) ? HPAGE_SIZE : PAGE_SIZE;
	start_va = va & ~(page_size - 1);
	next_page_va = start_va + page_size;
	if ((va + size - 1) < va ||
	    (va + size - 1) >= next_page_va) {
		TRACE("User buffer crossed page boundary.\n");
		result = -EINVAL; /* special error for 'spanned pages'? */
		goto va_overflow;
	}
	entry->size = page_size;

	/* Map the page onto the PCI bus. */
	entry->bus_addr = pci_map_page(tlr->pci_dev, entry->page,
				       va - start_va, page_size,
				       DMA_BIDIRECTIONAL);
	if (entry->bus_addr == 0) {
		result = -ENOMEM;
		goto map_failed;
	}

	*entry_out = entry;
	return 0;

 map_failed:
 va_overflow:
	page_cache_release(entry->page);
 get_user_pages_failed:
	kfree(entry);
	return result;
}

static void release_mem_buffer(struct tlr_pcie_dev *tlr,
			       struct tlr_page_list_entry *entry)
{
	pci_unmap_page(tlr->pci_dev, entry->bus_addr, entry->size,
		       DMA_BIDIRECTIONAL);
	page_cache_release(entry->page);
	kfree(entry);
}

static long tlr_barmem_unlocked_ioctl(struct file *filp,
				      unsigned int cmd, unsigned long arg)
{
	struct tlr_pcie_dev *tlr = filp->private_data;
	unsigned long flags;

	switch (cmd) {
	case TILEPCI_IOC_MAP_TO_PCI_BUS:
	{
		/*
		 * This ioctl takes a userspace va and size and maps
		 * that page to the PCI bus.  We keep a linked list of
		 * mapped pages so we can release the mappings when
		 * the barmem device is closed.
		 */
		struct tilepci_bus_map_params params;
		struct tlr_page_list_entry *entry = NULL;
		int err;

		if (copy_from_user(&params, (void __user *)arg,
				   sizeof(params)))
			return -EFAULT;

		err = register_mem_buffer(tlr, params.va, params.size, &entry);
		if (err)
			return err;

		params.bus_addr = entry->bus_addr;
		if (copy_to_user((void __user *)arg, &params,
				 sizeof(params))) {
			release_mem_buffer(tlr, entry);
			return -EFAULT;
		}

		spin_lock_irqsave(&tlr->barmem_lock, flags);
		list_add_tail(&entry->list, &tlr->barmem_page_list);
		spin_unlock_irqrestore(&tlr->barmem_lock, flags);

		return 0;
	}

	default:
		return -ENOTTY;
	}
}

#if LINUX_VERSION_CODE <= 0x020623  /* 2.6.35 */
static int tlr_barmem_ioctl(struct inode *inode, struct file *filp,
			    unsigned int cmd, unsigned long arg)
{
	return tlr_barmem_unlocked_ioctl(filp, cmd, arg);
}
#endif

static loff_t tlr_barmem_llseek(struct file *filp, loff_t off, int whence)
{
	long newpos;

	switch (whence) {
	case 0: /* SEEK_SET */
		newpos = off;
		break;

	case 1: /* SEEK_CUR */
		newpos = filp->f_pos + off;
		break;

	case 2: /* SEEK_END */
		/* We have no size, so this is the same as SEEK_SET. */
		newpos = off;
		break;

	default: /* can't happen */
		return -EINVAL;
	}
	if (newpos < 0)
		return -EINVAL;
	filp->f_pos = newpos;
	return newpos;
}


static int tlr_barmem_release(struct inode *inode, struct file *filp)
{
	struct tlr_pcie_dev *tlr = filp->private_data;
	unsigned long flags;
	struct tlr_page_list_entry *entry;
	struct list_head *ptr;
	struct list_head *next;

	spin_lock_irqsave(&tlr->barmem_lock, flags);
	tlr->barmem_count--;
	if (tlr->barmem_count == 0) {
		iounmap(tlr->barmem);

		/* Release all pages that the device used to reference. */
		list_for_each_safe(ptr, next, &tlr->barmem_page_list) {
			entry = list_entry(ptr, struct tlr_page_list_entry,
					   list);
			list_del(ptr);
			release_mem_buffer(tlr, entry);
		}
	}
	spin_unlock_irqrestore(&tlr->barmem_lock, flags);

	return 0;
}



static const struct file_operations tlr_barmem_ops = {
	.owner = THIS_MODULE,
	.llseek = tlr_barmem_llseek,
	.read = tlr_barmem_read,
	.write = tlr_barmem_write,
#if LINUX_VERSION_CODE > 0x020623  /* 2.6.35 */
	.unlocked_ioctl = tlr_barmem_unlocked_ioctl,
#else
	.ioctl = tlr_barmem_ioctl,
#endif
	.release = tlr_barmem_release,
};


static int tlr_barmem_open(struct inode *inode, struct file *filp)
{
	struct tlr_pcie_dev *tlr =
		container_of(inode->i_cdev, struct tlr_pcie_dev, cdev);
	unsigned long flags;

	filp->private_data = tlr;
	filp->f_op = &tlr_barmem_ops;

	/* Fail if the iBound isn't ready. */
	if (!tlr_is_ready(tlr))
		return -ENXIO;

	if (tlr_barmem_size(tlr) == 0) {
		ERR("BAR1 is not present.\n");
		return -ENXIO;
	}

	spin_lock_irqsave(&tlr->barmem_lock, flags);
	if (tlr->barmem_count == 0) {
		tlr->barmem = ioremap(tlr_barmem_base(tlr),
				      tlr_barmem_size(tlr));
		if (tlr->barmem == NULL) {
			spin_unlock_irqrestore(&tlr->barmem_lock, flags);
			ERR("Could not map BAR memory.\n");
			return -ENXIO;
		}
	}
	tlr->barmem_count++;
	spin_unlock_irqrestore(&tlr->barmem_lock, flags);

	return 0;
}


/**********************************************************************/
/*                        Chip Information File                       */
/**********************************************************************/


/* Calculate the link width in Gen-1 bandwidth equivalents.  For
 * example, a Gen-1 x4 link and a Gen-2 x2 link both return 4. */
static int calc_effective_link_width(struct pci_dev *dev)
{
	int pcie_caps_offset;
	int link_status_offset;
	u16 link_status;
	const int LINK_STATUS_OFFSET = 0x12;
	u32 width;
	u32 speed;

	/* Find the link status regs. */
	pcie_caps_offset = pci_find_capability(dev, PCI_CAP_ID_EXP);
	if (pcie_caps_offset == 0) {
		ERR("Could not read PCI-Express capability regs.\n");
		return 0;
	}

	/* Check whether we actually got at least the specified link BW. */
	link_status_offset = pcie_caps_offset + LINK_STATUS_OFFSET;
	pci_read_config_word(dev, link_status_offset, &link_status);
	speed = (link_status & 0xf);
	width = ((link_status >> 4) & 0x3f);

	return speed * width;
}

static int tlr_info_seq_show(struct seq_file *s, void *token)
{
	struct tlr_pcie_dev *tlr = s->private;
	resource_size_t bar1_size;

	seq_printf(s, "CHIP_VERSION %d\n", tlr->chip_version);
	seq_printf(s, "CHIP_WIDTH %d\n", tlr->chip_width);
	seq_printf(s, "CHIP_HEIGHT %d\n", tlr->chip_height);
	seq_printf(s, "HOST_LINK_INDEX %d\n", tlr->link_index);
	seq_printf(s, "LINK_WIDTH %d\n",
		   calc_effective_link_width(tlr->pci_dev));

	bar1_size = tlr_barmem_size(tlr);
	if (bar1_size) {
		seq_printf(s, "LINK_BAR1_SIZE %llu\n",
			   (unsigned long long)bar1_size);
		seq_printf(s, "LINK_BAR1_ADDRESS %#llx\n",
			   (unsigned long long)tlr_barmem_base(tlr));
	}

	return 0;
}

static const struct file_operations tlr_info_ops = {
	.owner          = THIS_MODULE,
	.read           = seq_read,
	.llseek         = seq_lseek,
	.release        = single_release,
};

static int tlr_info_open(struct tlr_pcie_dev *tlr, struct file *filp)
{
	filp->f_op = &tlr_info_ops;
	return single_open(filp, tlr_info_seq_show, tlr);
}


static void fill_chip_version(struct tlr_pcie_dev *tlr)
{
	switch (tlr->pci_dev->device) {
	case TILERA_TILE64_DEV_ID:
		tlr->chip_version = 0;
		break;
	case TILERA_TILE_PRO_DEV_ID:
		tlr->chip_version = 1;
		break;
	default:
		tlr->chip_version = -1;
	}
}

static void fill_chip_size(struct tlr_pcie_dev *tlr)
{
#define RSHIM_INTERFACE_DISABLE 0x880
#define RSHIM_INTERFACE_DISABLE_TILEPRO36_SIGNATURE 0x89

	tlr->chip_width = 8;
	tlr->chip_height = 8;

	if (tlr->rshim_regs) {
		u32 value = readl(tlr->rshim_regs + RSHIM_INTERFACE_DISABLE);
		if (value == RSHIM_INTERFACE_DISABLE_TILEPRO36_SIGNATURE) {
			tlr->chip_width = 6;
			tlr->chip_height = 6;
		}
	}
}

/**********************************************************************/
/*                             RShim Access                           */
/**********************************************************************/

static ssize_t rshim_read(struct file *filp, char __user *buf,
			  size_t count, loff_t *f_pos)
{
	struct tlr_pcie_dev *tlr = filp->private_data;
	loff_t pos = *f_pos;
	u32 value;

	if ((pos & 0x3) ||
	    (pos >= PCIE_RSHIM_WINDOW_SIZE) ||
	    (count != sizeof(u32)))
		return -EINVAL;

	value = readl(tlr->rshim_regs + pos);
	if (copy_to_user(buf, &value, count))
		return -EFAULT;

	return count;
}


static ssize_t rshim_write(struct file *filp, const char __user *buf,
			   size_t count, loff_t *f_pos)
{
	struct tlr_pcie_dev *tlr = filp->private_data;
	loff_t pos = *f_pos;
	u32 value;

	if ((pos & 0x3) ||
	    (pos >= PCIE_RSHIM_WINDOW_SIZE) ||
	    (count != sizeof(u32)))
		return -EINVAL;

	if (copy_from_user(&value, buf, count))
		return -EFAULT;

	writel(value, tlr->rshim_regs + pos);

	return count;
}


static const struct file_operations rshim_file_ops = {
	.owner          = THIS_MODULE,
	.read           = rshim_read,
	.write          = rshim_write,
};


static int tlr_rshim_open(struct tlr_pcie_dev *tlr, struct file *filp)
{
	filp->f_op = &rshim_file_ops;
	filp->private_data = tlr;

	if (tlr->chip_version == 0) {
		ERR("rshim access via PCIE is only supported on TilePro.\n");
		return -ENODEV;
	}

	return 0;
}


/**********************************************************************/
/*                   Module Loading and Device Probe                  */
/**********************************************************************/

static struct pci_device_id ids[] = {
	{ PCI_DEVICE(TILERA_VENDOR_ID, TILERA_TILE64_DEV_ID), },
	{ PCI_DEVICE(TILERA_VENDOR_ID, TILERA_TILE_PRO_DEV_ID), },
	{ 0, }
};
MODULE_DEVICE_TABLE(pci, ids);


static int tlr_generic_open(struct inode *inode, struct file *filp)
{
	unsigned int minor = MINOR(inode->i_rdev);
	int result;
	struct tlr_pcie_dev *tlr =
		container_of(inode->i_cdev, struct tlr_pcie_dev, cdev);

	if (minor == TILEPCI_BOOT_MINOR)
		result = tlr_boot_open(inode, filp);
	else if (minor == TILEPCI_INFO_MINOR)
		result = tlr_info_open(tlr, filp);
	else if (minor == TILEPCI_RSHIM_MINOR)
		result = tlr_rshim_open(tlr, filp);
	else if (minor == TILEPCI_BARMEM_MINOR)
		result = tlr_barmem_open(inode, filp);
	else
		result = -ENODEV;

	return result;
}


static const struct file_operations tlr_generic_ops = {
	.owner = THIS_MODULE,
	.open = tlr_generic_open,
};


static int tlr_add_device_nodes(struct tlr_pcie_dev *tlr)
{
	struct cdev *cdev = &tlr->cdev;

	/* Allocate some major/minor numbers. */
	dev_t first;
	int num_devs = NUM_MINOR_DEVICES;
	int err = alloc_chrdev_region(&first, 0, num_devs,
				      (char *)driver_name);
	if (err != 0)
		return err;

	/* Register the device. */
	cdev_init(cdev, &tlr_generic_ops);
	cdev->owner = THIS_MODULE;
	err = cdev_add(cdev, first, num_devs);
	if (err != 0) {
		unregister_chrdev_region(first, num_devs);
		return err;
	}
	tlr->first_dev = first;
	return 0;
}


static void tlr_remove_device_nodes(struct tlr_pcie_dev *tlr)
{
	cdev_del(&tlr->cdev);
	unregister_chrdev_region(tlr->first_dev, NUM_MINOR_DEVICES);
}


static int probe(struct pci_dev *pci_dev, const struct pci_device_id *id)
{
	static int link_index;
	int			 err;
	struct tlr_pcie_dev	*tlr;

	/* Our device can use 64-bit DMA addresses. */
	if (pci_set_dma_mask(pci_dev, DMA_BIT_MASK(64)))
		WARNING("Couldn't enable 64-bit DMA for device.\n");

	/* Get some memory for this device's driver state. */
	tlr = kmalloc(sizeof(*tlr), GFP_KERNEL);
	if (tlr == NULL)
		return -ENOMEM;
	memset(tlr, 0, sizeof(*tlr));

	/* Initialize TLR object */
	tlr->pci_dev = pci_dev;
	dev_set_drvdata(&pci_dev->dev, tlr);
	spin_lock_init(&tlr->barmem_lock);
	INIT_LIST_HEAD(&tlr->barmem_page_list);
	spin_lock_init(&tlr->is_ready_lock);
	mutex_init(&tlr->boot_mutex);
	tlr->link_index = link_index++;


	/* Enable the device. */
	err = pci_enable_device(pci_dev);
	if (err != 0)
		goto enable_failed;

	/* Map in the memory-mapped IO registers. */
	tlr->regs = ioremap(pci_resource_start(pci_dev, 0),
			    sizeof(struct pcie_host_mmio_regs));

	if (tlr->regs == NULL) {
		ERR("Failed to map Tilera MMIO regs\n");
		err = -ENOMEM;
		goto map_failed;
	}

	/* Figure out what kind of chip architecture it is. */
	fill_chip_version(tlr);

	/* Map in the rshim registers if TilePro. */
	if (tlr->chip_version == 1) {
		tlr->rshim_regs = ioremap(pci_resource_start(pci_dev, 0) +
					  PCIE_RSHIM_WINDOW_OFFSET,
					  PCIE_RSHIM_WINDOW_SIZE);
		if (tlr->rshim_regs == NULL) {
			ERR("Failed to map rshim registers\n");
			err = -ENOMEM;
			goto rshim_map_failed;
		}
	}

	fill_chip_size(tlr);

	/* Create our character and boot devices. */
	err = tlr_add_device_nodes(tlr);
	if (err != 0)
		goto cdev_failed;

	/* Try to connect to the tile-side driver if it is already
	 * running.  This might happen if the card booted from SROM or
	 * some other interface. */
	(void) tlr_is_pending(tlr);

	return 0;

 cdev_failed:
	if (tlr->rshim_regs)
		iounmap(tlr->rshim_regs);
 rshim_map_failed:
	iounmap(tlr->regs);
 map_failed:
	pci_disable_device(pci_dev);
 enable_failed:
	kfree(tlr);
	return err;
}


/* Called via pci_unregister_driver() when the module is removed. */
static void remove(struct pci_dev *pci_dev)
{
	struct tlr_pcie_dev *tlr = dev_get_drvdata(&pci_dev->dev);

	tlr_remove_device_nodes(tlr);

	iounmap(tlr->regs);
	if (tlr->rshim_regs)
		iounmap(tlr->rshim_regs);

	pci_disable_device(pci_dev);

	kfree(tlr);
	dev_set_drvdata(&pci_dev->dev, NULL);
}

static struct pci_driver pci_driver = {
	.name = "tilepci",
	.id_table = ids,
	.probe = probe,
	.remove = remove,
};

static int __init pci_skel_init(void)
{
	INFO("Loading %s\n", driver_name);
	return pci_register_driver(&pci_driver);
}

static void __exit pci_skel_exit(void)
{
	pci_unregister_driver(&pci_driver);
}

#ifdef inline
#undef inline
#endif	/* ifdef inline */

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Tilera Corp.");

module_init(pci_skel_init);

module_exit(pci_skel_exit);
