/*
 * Copyright 2011 Tilera Corporation. All Rights Reserved.
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation, version 2.
 *
 *   This program is distributed in the hope that it will be useful, but
 *   WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, GOOD TITLE or
 *   NON INFRINGEMENT.  See the GNU General Public License for
 *   more details.
 */

#include <linux/linkage.h>
#include <linux/unistd.h>
#include <asm/irqflags.h>
#include <asm/processor.h>
#include <arch/abi.h>
#include <arch/spr_def.h>

#ifdef __tilegx__
#define bnzt bnezt
#endif

STD_ENTRY(current_text_addr)
	{ move r0, lr; jrp lr }
	STD_ENDPROC(current_text_addr)

/*
 * Implement execve().  The i386 code has a note that forking from kernel
 * space results in no copy on write until the execve, so we should be
 * careful not to write to the stack here.
 */
STD_ENTRY(kernel_execve)
	moveli TREG_SYSCALL_NR_NAME, __NR_execve
	swint1
	jrp lr
	STD_ENDPROC(kernel_execve)

/* Delay a fixed number of cycles. */
STD_ENTRY(__delay)
	{ addi r0, r0, -1; bnzt r0, . }
	jrp lr
	STD_ENDPROC(__delay)

/*
 * We don't run this function directly, but instead copy it to a page
 * we map into every user process.  See vdso_setup().
 */
	.pushsection .data
ENTRY(__rt_sigreturn)
	moveli TREG_SYSCALL_NR_NAME,__NR_rt_sigreturn
	swint1
	ENDPROC(__rt_sigreturn)
	ENTRY(__rt_sigreturn_end)
	.popsection

STD_ENTRY(dump_stack)
	{ move r2, lr; lnk r1 }
	{ move r4, r52; addli r1, r1, dump_stack - . }
	{ move r3, sp; j _dump_stack }
	jrp lr   /* keep backtracer happy */
	STD_ENDPROC(dump_stack)

STD_ENTRY(KBacktraceIterator_init_current)
	{ move r2, lr; lnk r1 }
	{ move r4, r52; addli r1, r1, KBacktraceIterator_init_current - . }
	{ move r3, sp; j _KBacktraceIterator_init_current }
	jrp lr   /* keep backtracer happy */
	STD_ENDPROC(KBacktraceIterator_init_current)

/*
 * Reset our stack to r1/r2 (sp and ksp0+cpu respectively), then
 * free the old stack (passed in r0) and re-invoke cpu_idle().
 * We update sp and ksp0 simultaneously to avoid backtracer warnings.
 */
STD_ENTRY(cpu_idle_on_new_stack)
	{
	 move sp, r1
	 mtspr SPR_SYSTEM_SAVE_K_0, r2
	}
	jal free_thread_info
	j cpu_idle
	STD_ENDPROC(cpu_idle_on_new_stack)

/* Loop forever on a nap during SMP boot. */
STD_ENTRY(smp_nap)
	nap
	j smp_nap /* we are not architecturally guaranteed not to exit nap */
	jrp lr    /* clue in the backtracer */
	STD_ENDPROC(smp_nap)

/*
 * Enable interrupts racelessly and then nap until interrupted.
 * Architecturally, we are guaranteed that enabling interrupts via
 * mtspr to INTERRUPT_CRITICAL_SECTION only interrupts at the next PC.
 * This function's _cpu_idle_nap address is special; see intvec.S.
 * When interrupted at _cpu_idle_nap, we bump the PC forward 8, and
 * as a result return to the function that called _cpu_idle().
 */
STD_ENTRY(_cpu_idle)
	movei r1, 1
	mtspr INTERRUPT_CRITICAL_SECTION, r1
	IRQ_ENABLE(r2, r3)             /* unmask, but still with ICS set */
	mtspr INTERRUPT_CRITICAL_SECTION, zero
	.global _cpu_idle_nap
_cpu_idle_nap:
	nap
	jrp lr
	STD_ENDPROC(_cpu_idle)
#ifdef CONFIG_FEEDBACK_COLLECT

	/* Provide the header of the .feedback section. */
	.section .feedback.start, "aw"
	.align 8
	.global __feedback_section_start
__feedback_section_start:
	.word 0x4fd5adb1	/* FEEDBACK_HEADER_MAGIC */
	.word 1			/* FEEDBACK_HEADER_VERSION */

	.word 0
	.word __feedback_section_end - __feedback_section_start
	.word __feedback_functions_start - __feedback_section_start
	.word __feedback_functions_end - __feedback_section_start

	.global __feedback_edges_count
__feedback_edges_count:
	.word 0			/* ConflictMissGraph starts out empty. */
	.word __feedback_section_end - __feedback_section_start

	/* No support for the linker hooks generated by the compiler. */
	.section .text.__feedback_function_entered,"ax"
	.weak __feedback_function_entered
__feedback_function_entered:
	jrp lr
	ENDPROC(__feedback_function_entered)
	.weak __feedback_function_resumed
	.weak __feedback_function_entered_asm
__feedback_function_resumed = __feedback_function_entered
__feedback_function_entered_asm = __feedback_function_entered

	.section .text.__gcov_interval_profiler,"ax"
	.weak __gcov_interval_profiler
__gcov_interval_profiler:
	jrp lr
	ENDPROC(__gcov_interval_profiler)

	.section .text.__gcov_pow2_profiler,"ax"
	.weak __gcov_pow2_profiler
__gcov_pow2_profiler:
	jrp lr
	ENDPROC(__gcov_pow2_profiler)

	.section .text.__gcov_one_value_profiler,"ax"
	.weak __gcov_one_value_profiler
__gcov_one_value_profiler:
	jrp lr
	ENDPROC(__gcov_one_value_profiler)

	.section .text.__gcov_indirect_call_profiler,"ax"
	.weak __gcov_indirect_call_profiler
__gcov_indirect_call_profiler:
	jrp lr
	ENDPROC(__gcov_indirect_call_profiler)

	.section .text.__gcov_average_profiler,"ax"
	.weak __gcov_average_profiler
__gcov_average_profiler:
	jrp lr
	ENDPROC(__gcov_average_profiler)

	.section .text.__gcov_ior_profiler,"ax"
	.weak __gcov_ior_profiler
__gcov_ior_profiler:
	jrp lr
	ENDPROC(__gcov_ior_profiler)

#endif
