/*
 * Copyright 2011 Tilera Corporation. All Rights Reserved.
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation, version 2.
 *
 *   This program is distributed in the hope that it will be useful, but
 *   WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, GOOD TITLE or
 *   NON INFRINGEMENT.  See the GNU General Public License for
 *   more details.
 *
 * Tilera-specific I2C driver.
 *
 * This source code is derived from the following driver:
 *
 * i2c Support for Atmel's AT91 Two-Wire Interface (TWI)
 *
 * Copyright (C) 2004 Rick Bronson
 * Converted to 2.6 by Andrew Victor <andrew@sanpeople.com>
 *
 * Borrowed heavily from original work by:
 * Copyright (C) 2000 Philip Edelbrock <phil@stimpy.netroedge.com>
 */

#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/module.h>
#include <linux/err.h>
#include <linux/i2c.h>
#include <linux/errno.h>
#include <linux/platform_device.h>
#include <linux/uaccess.h>
#include <hv/hypervisor.h>
#include <hv/drv_i2cm_intf.h>

#define DRV_NAME	"i2c-tile"

static char i2cm_device[16] = "i2cm/0";

/* Handle for hypervisor device. */
static int i2cm_hv_devhdl;

static int xfer_msg(struct i2c_adapter *adap, struct i2c_msg *pmsg)
{
	int retval = 0;
	int data_offset = 0;
	int addr = pmsg->addr;
	int length = pmsg->len;
	char *buf = pmsg->buf;

	/* HV uses 8-bit slave addresses. */
	addr <<= 1;

	while (length) {
		int hv_retval;
		tile_i2c_addr_desc_t hv_offset = {{
			.addr = addr,
			.data_offset = data_offset,
		}};

		int bytes_this_pass = length;
		if (bytes_this_pass > HV_I2CM_CHUNK_SIZE)
			bytes_this_pass = HV_I2CM_CHUNK_SIZE;

		if (pmsg->flags & I2C_M_RD)
			hv_retval = hv_dev_pread(i2cm_hv_devhdl, 0,
					(HV_VirtAddr) buf,
					bytes_this_pass,
					hv_offset.word);
		else
			hv_retval = hv_dev_pwrite(i2cm_hv_devhdl, 0,
					(HV_VirtAddr) buf,
					bytes_this_pass,
					hv_offset.word);
		if (hv_retval < 0) {
			if (hv_retval == HV_ENODEV) {
				pr_err(DRV_NAME ": %s failed, invalid I2C"
					" address or access denied.\n",
					(pmsg->flags & I2C_M_RD) ?
					"hv_dev_pread" : "hv_dev_pwrite");
				retval = -ENODEV;
			} else
				retval = -EIO;
			break;
		}

		buf += hv_retval;
		data_offset += hv_retval;
		length -= hv_retval;
	}

	return retval;
}

/*
 * Generic I2C master transfer routine.
 */
static int tile_i2c_xfer(struct i2c_adapter *adap, struct i2c_msg *pmsg,
			 int num)
{
	int ret, i;

	for (i = 0; i < num; i++) {
		if (pmsg->len && pmsg->buf) {

			/* We don't support ten bit chip address. */
			if (pmsg->flags & I2C_M_TEN)
				return -EINVAL;

			ret = xfer_msg(adap, pmsg);
			if (ret)
				return ret;

			pmsg++;
		} else
			return -EINVAL;
	}

	return i;
}

/*
 * Return list of supported functionality.
 */
static u32 tile_i2c_functionality(struct i2c_adapter *adap)
{
	return I2C_FUNC_I2C;
}

static const struct i2c_algorithm tile_i2c_algorithm = {
	.master_xfer	= tile_i2c_xfer,
	.functionality	= tile_i2c_functionality,
};

/*
 * This routine is called to register all I2C devices that are connected to
 * the I2C bus. This should be done at arch_initcall time, before declaring
 * the I2C adapter. This function does the following:
 *
 * 1. Retrieve the I2C device list from the HV which selectively grants the
 *    access permission of individual I2C devices, and build an array of struct
 *    i2c_board_info.
 * 2. Statically declare these I2C devices by calling
 *    i2c_register_board_info().
 */
static int __init tile_i2c_dev_init(void)
{
	struct i2c_board_info *tile_i2c_devices;
	tile_i2c_desc_t *tile_i2c_desc;
	int i2c_desc_size;
	int i2c_devs = 0;
	int ret;
	int i;

	/* Open the HV i2cm device. */
	i2cm_hv_devhdl = hv_dev_open((HV_VirtAddr)i2cm_device, 0);
	if (i2cm_hv_devhdl < 0) {
		if (i2cm_hv_devhdl == HV_ENODEV)
			printk(KERN_DEBUG DRV_NAME ": no hv driver\n");
		else
			printk(KERN_WARNING DRV_NAME ": open failure: %d\n",
			       i2cm_hv_devhdl);
		return -EINVAL;
	}

	ret = hv_dev_pread(i2cm_hv_devhdl, 0, (HV_VirtAddr)&i2c_devs,
			sizeof(i2c_devs), I2C_GET_NUM_DEVS_OFF);
	if (ret <= 0) {
		pr_err(DRV_NAME ": hv_dev_pread(I2C_GET_NUM_DEVS_OFF)"
		       " failed, error %d\n", ret);
		return -EIO;
	}

	if (i2c_devs == 0)
		return 0;

	pr_info(DRV_NAME ": detected %d I2C devices.\n", i2c_devs);

	i2c_desc_size = i2c_devs * sizeof(tile_i2c_desc_t);
	tile_i2c_desc = kzalloc(i2c_desc_size, GFP_KERNEL);
	if (!tile_i2c_desc)
		return -ENOMEM;

	ret = hv_dev_pread(i2cm_hv_devhdl, 0, (HV_VirtAddr)tile_i2c_desc,
			i2c_desc_size, I2C_GET_DEV_INFO_OFF);
	if (ret <= 0) {
		pr_err(DRV_NAME ": hv_dev_pread(I2C_GET_DEV_INFO_OFF)"
		       " failed, error %d\n", ret);
		return -EIO;
	}

	i2c_desc_size = i2c_devs * sizeof(struct i2c_board_info);
	tile_i2c_devices = kzalloc(i2c_desc_size, GFP_KERNEL);
	if (!tile_i2c_devices)
		return -ENOMEM;

	for (i = 0; i < i2c_devs; i++) {
		strncpy(tile_i2c_devices[i].type, tile_i2c_desc[i].name,
			I2C_NAME_SIZE);
		/* HV uses 8-bit slave addresses, convert to 7bit for Linux. */
		tile_i2c_devices[i].addr = tile_i2c_desc[i].addr >> 1;
	}

	ret = i2c_register_board_info(0, tile_i2c_devices, i2c_devs);

	kfree(tile_i2c_desc);
	kfree(tile_i2c_devices);

	return ret;
}
arch_initcall(tile_i2c_dev_init);

/*
 * I2C adapter probe routine which registers the I2C adapter with the I2C core.
 */
static int __devinit tile_i2c_probe(struct platform_device *dev)
{
	struct i2c_adapter *adapter;
	int ret;

	adapter = kzalloc(sizeof(struct i2c_adapter), GFP_KERNEL);
	if (adapter == NULL) {
		ret = -ENOMEM;
		goto malloc_err;
	}

	adapter->owner   = THIS_MODULE;

	/*
	 * If "dev->id" is negative we consider it as zero.
	 * The reason to do so is to avoid sysfs names that only make
	 * sense when there are multiple adapters.
	 */
	adapter->nr = dev->id != -1 ? dev->id : 0;
	snprintf(adapter->name, sizeof(adapter->name), "%s.%u",
		 dev_name(&dev->dev), adapter->nr);

	adapter->algo = &tile_i2c_algorithm;
	adapter->class = I2C_CLASS_HWMON;
	adapter->dev.parent = &dev->dev;

	ret = i2c_add_numbered_adapter(adapter);
	if (ret < 0) {
		dev_err(&dev->dev, "registration failed\n");
		goto add_adapter_err;
	}

	platform_set_drvdata(dev, adapter);

	return 0;

add_adapter_err:
	kfree(adapter);
malloc_err:
	return ret;
}

/*
 * I2C adapter cleanup routine.
 */
static int __devexit tile_i2c_remove(struct platform_device *dev)
{
	struct i2c_adapter *adapter = platform_get_drvdata(dev);
	int rc;

	rc = i2c_del_adapter(adapter);
	platform_set_drvdata(dev, NULL);

	kfree(adapter);

	return rc;
}

static struct platform_driver tile_i2c_driver = {
	.driver		= {
		.name	= DRV_NAME,
		.owner	= THIS_MODULE,
	},
	.probe		= tile_i2c_probe,
	.remove		= __devexit_p(tile_i2c_remove),
};

/*
 * Driver init routine.
 */
static int __init tile_i2c_init(void)
{
	struct platform_device *dev;
	int err;

	err = platform_driver_register(&tile_i2c_driver);
	if (err)
		return err;

	dev = platform_device_register_simple(DRV_NAME, -1, NULL, 0);
	if (IS_ERR(dev)) {
		err = PTR_ERR(dev);
		pr_warning("i2c-tile init failed: %d\n", err);
		goto unreg_platform_driver;
	}

	return 0;

unreg_platform_driver:
	platform_driver_unregister(&tile_i2c_driver);
	return err;
}

/*
 * Driver cleanup routine.
 */
static void __exit tile_i2c_exit(void)
{
	platform_driver_unregister(&tile_i2c_driver);
}

module_init(tile_i2c_init);
module_exit(tile_i2c_exit);
