/*
 * Copyright 2011 Tilera Corporation. All Rights Reserved.
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation, version 2.
 *
 *   This program is distributed in the hope that it will be useful, but
 *   WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, GOOD TITLE or
 *   NON INFRINGEMENT.  See the GNU General Public License for
 *   more details.
 *
 * Routines for mapping memory into BAR1.
 */
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/cdev.h>
#include <linux/mm.h>
#include <linux/uaccess.h>
#include <linux/semaphore.h>
#include <linux/pagemap.h>
#include <linux/pci.h>
#include <linux/percpu.h>
#include <linux/hugetlb.h>

#include <asm/tilepci.h>
#include <asm/cacheflush.h>

#include "tilepci_endp.h"



/*
 * Handle an iomem registration request by getting the backing pages
 * and registering each of them with the hypervisor driver.  Unlike
 * the xgbe and direct-to-HV ZC APIs, these pages are registered at
 * the dedicated tile, so they do not need to be registered on every
 * tile.  Any attempt to map at an offset that already has a page
 * mapped will fail.
 *
 * We track each mapped page with the file that was used to create it;
 * when that file unmaps a region or is closed, we'll close only its
 * mappings.
 *
 * This is yet another variant of the code in xgbe.c.
 */
static int map_bar_pages(struct tlr_pcie_dev *tlr, tilepci_barmem_map_t *map,
			 struct file *filp)
{
	struct mm_struct *mm = current->mm;
	int i;

	/* Sanity check the VA, size, and offset. */
	if (((unsigned long)map->va & (HPAGE_SIZE - 1)) ||
	    (map->size & (HPAGE_SIZE - 1)) ||
	    (map->offset & (HPAGE_SIZE - 1)) ||
	    map->va + map->size <= map->va) {
		return -EINVAL;
	}

	/* Synchronize access to the mapped pages list. */
	if (down_interruptible(&tlr->bar1_pages_mutex))
		return -ERESTARTSYS;

	/*
	 * Make sure that none of the relevant offsets are already
	 * mapped.
	 */
	for (i = map->offset >> HPAGE_SHIFT;
	     i < ((map->offset + map->size) >> HPAGE_SHIFT);
	     i++) {
		if (tlr->bar1_pages[i].page) {
			up(&tlr->bar1_pages_mutex);
			return -EADDRINUSE;
		}
	}


	/*
	 * We use get_user_pages to grab and lock the pages, so we need
	 * this semaphore.
	 */
	down_read(&mm->mmap_sem);

	/*
	 * Get and register our pages, one at a time.
	 */
	for (i = 0; i < (map->size >> HPAGE_SHIFT); i++) {
		int index = (map->offset >> HPAGE_SHIFT) + i;
		struct vm_area_struct *vma;
		struct page *page;
		int retval;

		/* First get the page. */
		unsigned long va =
			(unsigned long)(map->va) + (i << HPAGE_SHIFT);
		struct page *temp_pages[1];
		retval = get_user_pages(current, mm, va,
					1, 1, 0, temp_pages,
					&vma);
		page = temp_pages[0];

		if (retval < 0)
			retval = -EFAULT;
		else if (retval != 1 || !is_vm_hugetlb_page(vma)) {
			/* If we didn't get a large page, we fail. */
			retval = -EINVAL;
		} else {
			/* We got a page, see if we can register it. */
			HV_PhysAddr pa =
				((HV_PhysAddr)page_to_pfn(page)) << PAGE_SHIFT;
			struct pcie_barmem_config config = {
				.is_map = 1,
				.bar = 1,
				.hpage_pa = pa,
				.offset = index << HPAGE_SHIFT
			};

			int err = hv_dev_pwrite(
				tlr->hv_channel_ctl_fd, 0,
				(HV_VirtAddr)&config, sizeof(config),
				PCIE_BARMEM_CONFIG_OFF);
			if (err == sizeof(config)) {
				tlr->bar1_pages[index].page = page;
				tlr->bar1_pages[index].filp = filp;
			} else {
				retval = -EIO; /* Who knows what happened! */
			}
		}

		/*
		 * If the last page failed, we need to back out everything
		 * we've done so far, and return an error.
		 */
		if (retval < 0) {
			int j;
			if (page)
				page_cache_release(page);

			/*
			 * Unregister and release everything we've
			 * successfully registered up to this
			 * point.
			 */
			for (j = 0; j < i; j++) {
				int jindex = (map->offset >> HPAGE_SHIFT) + j;
				struct pcie_barmem_config unreg = {
					.is_map = 0,
					.bar = 1,
					.offset = jindex << HPAGE_SHIFT
				};
				page = tlr->bar1_pages[jindex].page;
				hv_dev_pwrite(
					tlr->hv_channel_ctl_fd, 0,
					(HV_VirtAddr)&unreg, sizeof(unreg),
					PCIE_BARMEM_CONFIG_OFF);

				page_cache_release(page);
				tlr->bar1_pages[jindex].page = NULL;
				tlr->bar1_pages[jindex].filp = NULL;
			}

			up_read(&mm->mmap_sem);
			up(&tlr->bar1_pages_mutex);

			return retval;
		}
	}

	up_read(&mm->mmap_sem);
	up(&tlr->bar1_pages_mutex);
	return 0;
}

/*
 * Unregister a range of pages.  Only pages that were mapped via the
 * specified file will be unmapped; if some other file made a mapping
 * within this range of offsets then it needs to be unregistered by
 * that file.
 */
static int unmap_bar_pages(struct tlr_pcie_dev *tlr, tilepci_barmem_map_t *map,
			 struct file *filp)
{
	int i;

	/* Round up to huge page size and sanity check. */
	map->size = (map->size + HPAGE_SIZE - 1) & ~(HPAGE_SIZE - 1);
	if ((map->offset & (HPAGE_SIZE - 1)) ||
	    map->offset + map->size <= map->offset) {
		return -EINVAL;
	}

	/* Synchronize access to the mapped pages list. */
	if (down_interruptible(&tlr->bar1_pages_mutex))
		return -ERESTARTSYS;

	/* Remove mappings for this file's mappings in that range. */
	for (i = map->offset >> HPAGE_SHIFT;
	     i < ((map->offset + map->size) >> HPAGE_SHIFT);
	     i++) {
		if (tlr->bar1_pages[i].page &&
		    tlr->bar1_pages[i].filp == filp) {
			/* First, remove the HV mapping. */
			struct pcie_barmem_config unreg = {
				.is_map = 0,
				.bar = 1,
				.offset = i << HPAGE_SHIFT,
			};
			hv_dev_pwrite(
				tlr->hv_channel_ctl_fd, 0,
				(HV_VirtAddr)&unreg, sizeof(unreg),
				PCIE_BARMEM_CONFIG_OFF);

			/* Then, release Linux ref cnt. */
			page_cache_release(tlr->bar1_pages[i].page);
			tlr->bar1_pages[i].page = NULL;
			tlr->bar1_pages[i].filp = NULL;
		}
	}

	up(&tlr->bar1_pages_mutex);
	return 0;
}

/* Unregister all the pages that the specified file has mapped. */
static int unmap_all_bar_pages(struct tlr_pcie_dev *tlr, struct file *filp)
{
	int i;

	/* Synchronize access to the mapped pages list. */
	if (down_interruptible(&tlr->bar1_pages_mutex))
		return -ERESTARTSYS;

	/* Remove all mappings for this file. */
	for (i = 0; i < PCIE_BAR1_PAGES; i++) {
		if (tlr->bar1_pages[i].page &&
		    tlr->bar1_pages[i].filp == filp) {
			/* First, remove the HV mapping. */
			struct pcie_barmem_config unreg = {
				.is_map = 0,
				.bar = 1,
				.offset = i << HPAGE_SHIFT,
			};
			hv_dev_pwrite(
				tlr->hv_channel_ctl_fd, 0,
				(HV_VirtAddr)&unreg, sizeof(unreg),
				PCIE_BARMEM_CONFIG_OFF);

			/* Then, release Linux ref cnt. */
			page_cache_release(tlr->bar1_pages[i].page);
			tlr->bar1_pages[i].page = NULL;
			tlr->bar1_pages[i].filp = NULL;
		}
	}

	up(&tlr->bar1_pages_mutex);
	return 0;
}


/*
 * System call handlers that allow the user to map and unmap huge page
 * VA ranges into the BAR1 window.
 */

static long tlr_barmem_ioctl(struct file *filp,
			     unsigned int cmd, unsigned long arg)
{
	struct tlr_pcie_dev *tlr = filp->private_data;
	tilepci_barmem_map_t map;

	switch (cmd) {
	case TILEPCI_IOC_BARMEM_MAP:
	{
		if (copy_from_user(&map, (void __user *)arg,
				   sizeof(tilepci_barmem_map_t)))
			return -EFAULT;

		return map_bar_pages(tlr, &map, filp);
	}

	case TILEPCI_IOC_BARMEM_UNMAP:
	{
		if (copy_from_user(&map, (void __user *)arg,
				   sizeof(tilepci_barmem_map_t)))
			return -EFAULT;

		return unmap_bar_pages(tlr, &map, filp);
	}

	default:
		return -EINVAL;
	}

	return 0;
}

/* Device release routine; unregisters all the pages mapped via this file. */
static int tlr_barmem_release(struct inode *inode, struct file *filp)
{
	struct tlr_pcie_dev *tlr = filp->private_data;

	return unmap_all_bar_pages(tlr, filp);
}


static const struct file_operations tlr_barmem_ops = {
	.owner = THIS_MODULE,
	.unlocked_ioctl = tlr_barmem_ioctl,
	.release = tlr_barmem_release,
};

int tlr_barmem_open(struct tlr_pcie_dev *tlr, struct file *filp)
{
	filp->private_data = tlr;
	filp->f_op = &tlr_barmem_ops;

	if (!tlr_is_ready(tlr))
		return -ENXIO;

	return 0;
}
