/*
 * Copyright 2011 Tilera Corporation. All Rights Reserved.
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation, version 2.
 *
 *   This program is distributed in the hope that it will be useful, but
 *   WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, GOOD TITLE or
 *   NON INFRINGEMENT.  See the GNU General Public License for
 *   more details.
 *
 * Defines the kernel-to-userspace API for access to
 * the Bare Metal Environment.
 */

/**
 * @file bme.h
 *
 * @addtogroup bmelinux
 * @{
 */

#ifndef _ASM_TILE_BME_H
#define _ASM_TILE_BME_H

#ifndef __KERNEL__
#include <stdint.h>
#else
#include <linux/types.h>
#endif

#include <linux/ioctl.h>

/* A unique ioctl prefix for the BME memory driver. */
#ifndef __DOXYGEN__
#define BME_IOC_TYPE 0xB6
#endif


/* BME User Space API */


/**
 * Descriptor for user memory attributes.
 */
struct bme_user_mem_desc {
	void __user *va;	/**< Address of memory. */
	uint32_t len;		/**< Length of memory in bytes. */
};

/**
 * Descriptor for physical memory attributes.
 */
struct bme_phys_mem_desc {
	uint64_t pa;		/**< Physical address of memory. */
	uint32_t len;		/**< Length of memory in bytes. */
	uint64_t pte;		/**< Caching attributes. */
};

/**
 * Describes pages underlying user memory.
 */
struct bme_user_mem_desc_io {
	/** Describes the address range of the user memory. */
	struct bme_user_mem_desc user;
	/** Return value for number of pages underlying the user memory. */
	uint32_t num_pages;
};

/**
 * Struct to lock and describe user memory.
 */
struct bme_phys_mem_desc_io {
	/** Describes the address range of the user memory. */
	struct bme_user_mem_desc user;
	/** Specifies number of pages underlying this address range. */
	uint32_t num_pages;
	/** Return info for physical characteristics of memory.
	 * Must point to num_pages * sizeof(struct bme_phys_mem_desc) bytes. */
	struct bme_phys_mem_desc *phys;
	/** 0 if results are not valid, non-zero if all	of the requested
	 * memory was locked. */
	int results_are_valid;
};

/** Get the number of pages this range of memory covers. */
#define BME_IOC_GET_NUM_PAGES _IO(BME_IOC_TYPE, 0x0)

/**
 * Lock the memory so it can be accessed by BME tiles.  User must provide
 * space for the number of pages included in this range.  That number may
 * be obtained by BME_IOC_GET_NUM_PAGES, above.
 */
#define BME_IOC_LOCK_MEMORY _IO(BME_IOC_TYPE, 0x1)

#endif

/** @} */
