/*
 * Copyright 2011 Tilera Corporation. All Rights Reserved.
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation, version 2.
 *
 *   This program is distributed in the hope that it will be useful, but
 *   WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, GOOD TITLE or
 *   NON INFRINGEMENT.  See the GNU General Public License for
 *   more details.
 */

/**
 * @file drv_gpio_intf.h
 * Interface definitions for the gpio driver.
 *
 * The Hypervisor GPIO driver provides a manner for supervisors to
 * efficiently access the FlexIO/GPIO pins of Tilera chips.
 * The basic interface to the GPIO driver is via the open, pread, and pwrite
 * supervisor/hypervisor interface calls.
 *
 * The GPIO driver provides two primary manners to access the GPIO pins,
 * via direct IO manipulation and via small programs loaded into the GPIO
 * driver which then can maniputalte the GPIO pins.  By allowing two
 * alternative methods, simple pin manipulation can occur inside of the
 * supervisor, while performance critical pin manipulation can occur
 * inside of small interpreted programs which are loaded into the hypervisor
 * by a supervisor.  The interpreted programs are sandboxed inside of the
 * hypervisor to prevent the supervisor from executing arbitrary code
 * with the privileges of the hypervisor.
 *
 * The GPIO driver presents five distinct memory regions to the supervisor.
 * There is 1K worth of program store which begins at
 * GPIO_CTL_PROGRAM_STORE_OFF.  Next at offset GPIO_CTL_STATIC_DATA_OFF, there
 * is 1K of data reserved as scratchpad space for the programs.  This data is
 * readable and writable by the supervisor.  Next is GPIO_CTL_WRITE_OFF which
 * is 1K of data which is how the supervisor passes parameters to the
 * hypervisor.  These are the input parameters for a program executing in the
 * hypervisor.  Also, if offset GPIO_CTL_WRITE_OFF is written, this will kick
 * off a program to execute beginning at uint32_t value written to the first
 * word.  This program value must be in the GPIO_CTL_PROGRAM_STORE_OFF range.
 * At offset GPIO_CTL_READ_OFF, there is a 1KB region which is dedicated to
 * the return value returned from executing GPIO programs.  The return data
 * can be quite large to handle cases where buffers are returned.
 *
 * The config region GPIO_CTL_CONFIG_OFF contains configuration control for
 * the GPIO pins.  At the beginning of this range is a read/write mapping
 * of the register file from the downloadable programs.  There are
 * GPIO_NUMBER_OF_REGISTERS register each 32 bits in size.  This config
 * space also controls the data being driven to the pins in a slow manner
 * via GPIO_CTL_CONFIG_DATA_OFF.  Likewise the pin state can be read.  The
 * pins can be released to be inputs as seen by the Tilera chip by writing to
 * GPIO_CTL_CONFIG_DIR_OFF.  Also, the electrical drive strength can be
 * modified via GPIO_CTL_CONFIG_ELEC_OFF.  Please read the GPIO documentation
 * when modifying these registers as misconfiguration may case shorts if
 * incorrect directions are driven.
 *
 * The driver supports a single IRQ pin.  If this pin asserts, the
 * driver will pass an IRQ into the supervisor.  The supervisor can
 * configure which GPIO pin generates an IRQ by writing to
 * GPIO_CTL_IRQ_PIN_OFF.  The supervisor can obtain the IRQ number
 * assigned to that pin by reading GPIO_CTL_IRQ_NUMBER_OFF.
 *
 * The GPIO has GPIO_MAX_NUMBER_OF_DATA_BANKS banks on a nominal configuration.
 * This driver does not handle multiplexing of pins, and assigns all of the
 * GPIO pins to one supervisor.  In order to access the GPIO, open the first
 * GPIO device and this gives access to the GPIO pins through a banked model.
 *
 */

#ifndef _SYS_HV_DRV_GPIO_INTF_H
#define _SYS_HV_DRV_GPIO_INTF_H

/** Number of registers in simulated GPIO processor */
#define GPIO_NUMBER_OF_REGISTERS    256
/** There are four banks of 16 pins of GPIO on the TILE64 */
#define GPIO_MAX_NUMBER_OF_DATA_BANKS 4

/* Offsets within control device */


#define GPIO_CTL_PROGRAM_STORE_OFF  0    /**< Buffer that holds the
                                                little programs.  This is 1K
                                                and is managed by the OS */

#define GPIO_CTL_PROGRAM_STORE_END  (GPIO_CTL_PROGRAM_STORE_OFF+1023)
                                         /**< End of program store */

#define GPIO_CTL_STATIC_DATA_OFF    0x400   /**< Static data that is shared
                                                between the channel programs
                                                and the OS.  This is primarily
                                                for static data for the
                                                application */

#define GPIO_CTL_STATIC_DATA_END    (GPIO_CTL_STATIC_DATA_OFF+1023)
                                         /**< End of static data */

#define GPIO_CTL_WRITE_OFF          0x800   /**< Write to kick off a program.
                                                the first value is the program
                                                that wants to be executed and
                                                the rest is considered
                                                parameters to the control
                                                programs.  Reading this data
                                                has undefined results*/

#define GPIO_CTL_WRITE_END          (GPIO_CTL_WRITE_OFF+1023)
                                         /**< End of parameter region */

#define GPIO_CTL_READ_OFF           0xc00   /**< Read to get return values
                                                from last transaction.  Note
                                                that the first word returned
                                                should be the return code from
                                                the operation */

#define GPIO_CTL_READ_END         (GPIO_CTL_READ_OFF+1023)
                                         /**< End of return value region */

#define GPIO_CTL_CONFIG_BLOCK_SIZE  0x800
                                         /**< Size of configuration block */

#define GPIO_CTL_CONFIG_OFF       0x1000   /**< Config data, Read and write
                                                buffer */

#define GPIO_CTL_CONFIG_END       (GPIO_CTL_CONFIG_OFF+ \
                                    GPIO_CTL_CONFIG_BLOCK_SIZE-1)
                                         /**< End of config data region */

#define GPIO_CTL_CONFIG_RF_OFF    (GPIO_CTL_CONFIG_OFF + 0) /**< A way to
                                              access the register file */

#define GPIO_CTL_CONFIG_RF_END    (GPIO_CTL_CONFIG_RF_OFF + \
                                      (sizeof(uint32_t) * \
                                      GPIO_NUMBER_OF_REGISTERS) - 1)
                                         /**< End of register file region */

/** GPIO pin data (write) */
typedef struct
{
  uint16_t data;            /**< Data to write */
  uint16_t must_be_zero_0;  /**< Must be zero */
  uint16_t mask;            /**< Mask of pins to modify */
  uint16_t must_be_zero_1;  /**< Must be zero */
}
gpio_pin_data_t;

/** GPIO pin data (read) */
typedef struct
{
  uint16_t data;            /**< Data to read */
  uint16_t must_be_zero_0;  /**< Zero, ignore */
}
gpio_pin_data_read_t;

#define GPIO_CTL_CONFIG_DATA_OFF  (GPIO_CTL_CONFIG_RF_END + 1) /**< Direct
                                      access to the pins first word
                                      sets pins, second is a mask.
                                      When read, read only first word */

#define GPIO_CTL_CONFIG_DATA_END  (GPIO_CTL_CONFIG_DATA_OFF + \
                                    (sizeof(gpio_pin_data_t) * \
                                    GPIO_MAX_NUMBER_OF_DATA_BANKS)-1)
                                  /**< End of pin direct access region */

/** GPIO pin release (tristate) */
typedef struct
{
  uint16_t release;         /**< Pins to release */
  uint16_t must_be_zero_1;  /**< Must be zero */
}
gpio_pin_release_t;

#define GPIO_CTL_CONFIG_DIR_OFF   (GPIO_CTL_CONFIG_DATA_END+1) /**< Releases
                                    the pin direction, write one
                                    word at a time */

#define GPIO_CTL_CONFIG_DIR_END   (GPIO_CTL_CONFIG_DIR_OFF + \
                                    (GPIO_MAX_NUMBER_OF_DATA_BANKS * \
                                    sizeof(gpio_pin_release_t)) - 1)
                                  /**< End of pin release region */

#define GPIO_NUM_OF_ELEC_CONTROL_WORDS  4  /**< Number of electrical control
                                                words */
/** GPIO electrical control */
typedef struct
{
  uint32_t control[GPIO_NUM_OF_ELEC_CONTROL_WORDS];  /**< Electrical control */
}
gpio_pin_elect_control_t;

#define GPIO_CTL_CONFIG_ELEC_OFF  (GPIO_CTL_CONFIG_DIR_END + 1) /**< Sets up
                                          the electricals there is
                                          a byte per pin see gpio uarch
                                          documentation */

#define GPIO_CTL_CONFIG_ELEC_END  (GPIO_CTL_CONFIG_ELEC_OFF + \
                                    (GPIO_MAX_NUMBER_OF_DATA_BANKS * \
                                    sizeof(gpio_pin_elect_control_t))-1)
                                  /**< End of electrical control region */

/** GPIO IRQ pin location, to be written to the driver. */
typedef struct
{
  uint32_t pin_bank;       /**< Pin bank */
  uint32_t pin_location;   /**< Pin location */
}
gpio_irq_pin_t;

#define GPIO_CTL_IRQ_PIN_OFF (GPIO_CTL_CONFIG_ELEC_END + 1) /**< sets
                                    the IRQ callback pin parameters
                                            data should be
                                            [0] pin bank
                                            [1] pin location */

#define GPIO_CTL_IRQ_PIN_END (GPIO_CTL_IRQ_PIN_OFF + \
                                    sizeof(gpio_irq_pin_t) - 1)
                                  /**< End of IRQ callback region */

/** GPIO IRQ number, to be read from the driver. */
typedef struct
{
  uint32_t irq_number;     /**< IRQ downcall index. */
}
gpio_irq_number_t;

/** Offset from which to read the IRQ number. */
#define GPIO_CTL_IRQ_NUMBER_OFF (GPIO_CTL_IRQ_PIN_END + 1)

/** End of IRQ number region. */
#define GPIO_CTL_IRQ_NUMBER_END \
  (GPIO_CTL_IRQ_NUMBER_OFF + sizeof(gpio_irq_number_t) - 1)


/* Board wiring codes. */

#define GPIO_BOARD_WIRING_UNKNOWN      0 /**< Unknown GPIO wiring */
#define GPIO_BOARD_WIRING_IDE_REV1     1 /**< Rev 1 PCI-E board IDE wiring */
#define GPIO_BOARD_WIRING_IDE_REV2     2 /**< Rev 2+ PCI-E board IDE wiring */
#define GPIO_BOARD_WIRING_IDE_TE20G_V2 3 /**< Madison Rev 2 board IDE wiring */
#define GPIO_BOARD_WIRING_IDE_AMC_V1   4 /**< AMC Rev 1 board IDE wiring */
#define GPIO_BOARD_WIRING_IDE_BLADE_V1 5 /**< Blade Rev 1 board IDE wiring */
#define GPIO_BOARD_WIRING_IDE_BLADE_V2 6 /**< Blade Rev 2 board IDE wiring */

#define GPIO_CTL_BOARD_WIRING_CHARS 4 /**< Size of wiring code */

#define GPIO_CTL_BOARD_WIRING_OFF   (GPIO_CTL_IRQ_NUMBER_END + 1) /**< Returns
                                    the board wiring code as a word */

#define GPIO_CTL_BOARD_WIRING_END   (GPIO_CTL_BOARD_WIRING_OFF + \
                                     (GPIO_CTL_BOARD_WIRING_CHARS * \
                                     sizeof(char)))
                                 /**< End of board wiring code */

#define GPIO_MAX_INSTRUCTIONS_PER_CALL 16384 /**< Maximum of instructions
                                                    in control program before
                                                    the program is killed.
                                                    This prevents a program
                                                    from executing forever in
                                                    the hypervisor */

/* Instructions look like this

   opcode goes in lowest ordered byte
   for everything but BZ, BNZ, LI and LI_HIGH
   8 bits opcode
   8 bits dest operand
   8 bits src_a operand
   8 bits src_b operand

   for BZ, BNZ, LI and OUI
   8 bits opcode
   8 bits src/dest operand
   16 bits immediate

*/

/** A GPIO processor instruction. */
typedef struct
{
  uint8_t opcode;            /**< Operation code */
  uint8_t dest_operand;      /**< Destination operand */
  union {
    uint16_t immediate;      /**< Immediate value */
    struct {
      uint8_t src_a_operand; /**< First source operand */
      uint8_t src_b_operand; /**< Second source operand */
    };
  };
}
gpio_instruction_t;

/* Define the IOP instructions here. */

/** GPIO_SET_PINS sets pins of bank dest_operand to data in src_a_operand
  with a mask of src_b_operand.  Non-immediate. */
#define GPIO_SET_PINS   (0)

/** GPIO_RELEASE_PINS releases the direction of the pins (floats) those
  in bank dest_operand as pointed to by src_a_operand */
#define GPIO_RELEASE_PINS (1)

/** GPIO_READ_PINS Returns the pin data in bank src_a_operand into
  dest_operand */
#define GPIO_READ_PINS  (2)

/** GPIO_WAIT waits some time as scaled by src_a_operand */
#define GPIO_WAIT       (3)

/** GPIO_LW loads 32-bits worth of data as pointed to by src_a_operand into
  dest_operand.  The address is zero based at GPIO_CTL_PROGRAM_STORE_OFF */
#define GPIO_LW         (4)

/** GPIO_LH loads 16-bits worth of data as pointed to by src_a_operand into
  dest_operand.  The address is zero based at GPIO_CTL_PROGRAM_STORE_OFF */
#define GPIO_LH         (5)

/** GPIO_LB loads 8-bits worth of data as pointed to by src_a_operand into
  dest_operand.  The address is zero based at GPIO_CTL_PROGRAM_STORE_OFF */
#define GPIO_LB         (6)

/** GPIO_SW stores 32-bits worth of data in src_a_operand into the data
  addressed by dest_operand.  The address is zero based at
  GPIO_CTL_PROGRAM_STORE_OFF */
#define GPIO_SW         (7)

/** GPIO_SH stores 16-bits worth of data in src_a_operand into the data
  addressed by dest_operand.  The address is zero based at
  GPIO_CTL_PROGRAM_STORE_OFF */
#define GPIO_SH         (8)

/** GPIO_SB stores 8-bits worth of data in src_a_operand into the data
  addressed by dest_operand.  The address is zero based at
  GPIO_CTL_PROGRAM_STORE_OFF */
#define GPIO_SB         (9)

/* The arithmetic instructions GPIO_ADD, GPIO_SUB, GPIO_AND, GPIO_OR, GPIO_NOR
  GPIO_NOR, GPIO_SHL, GPIO_SHR take src_a_operand and src_b_operand inputs
  and store the results in dest_operand */
#define GPIO_ADD        (10)  /**< Add */
#define GPIO_SUB        (11)  /**< Subtract */
#define GPIO_AND        (12)  /**< Logical AND */
#define GPIO_OR         (13)  /**< Logical OR */
#define GPIO_NOR        (14)  /**< Logical NOR */
#define GPIO_SHL        (15)  /**< Shift left */
#define GPIO_SHR        (16)  /**< Shift right */

/** GPIO_LI loads immediate into dest_operand */
#define GPIO_LI         (17)

/** GPIO_OUI shifts immediate left by 16 and ORs the data with dest_operand
  with the result in dest_operand */
#define GPIO_OUI        (18)

/** GPIO_BZ branch if dest_operand is zero.  This is a sign extended relative
  branch of immediate */
#define GPIO_BZ         (19)

/** GPIO_BNZ branch if dest_operand is not zero.  This is a sign extended
  relative branch of immediate */
#define GPIO_BNZ        (20)

/** GPIO_JR jumps to the program address stored in src_a_operand */
#define GPIO_JR         (21)

/** GPIO_PRINT prints out the value of src_a_operand to the hypervisor debug
  console */
#define GPIO_PRINT      (22)

/** GPIO_DONE indicates to return control back to supervisor.  This must
  be executed at the end of all programs */
#define GPIO_DONE       (23)

#endif /* _SYS_HV_DRV_GPIO_INTF_H */
