/*
 * Copyright 2011 Tilera Corporation. All Rights Reserved.
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation, version 2.
 *
 *   This program is distributed in the hope that it will be useful, but
 *   WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, GOOD TITLE or
 *   NON INFRINGEMENT.  See the GNU General Public License for
 *   more details.
 */

/**
 * @file
 *
 * Declarations that are shared between various PCIE device models.
 */
#ifndef _SYS_HV_DRV_PCIE_COMMON_H
#define _SYS_HV_DRV_PCIE_COMMON_H

#ifndef __ASSEMBLER__

#ifdef __KERNEL__
#include <linux/stddef.h>
#else
#include <stddef.h>
#endif

#ifdef __DOXYGEN_TILEPCI_USER_API__
/** @ingroup direct_hv
 * Error codes that may be returned by the direct-to-hypervisor PCI
 * zero-copy command posting APIs. Such functions return 0 on success
 * and a negative value if an error occurs.
 *
 * In cases where a tilepci function failed due to a error reported by
 * system libraries, the error code will be the negation of the system
 * errno at the time of failure.  The @ref tilepci_strerror() function
 * will deliver error strings for both tilepci and system error codes.
 */
#else
/** Error codes for the direct-to-hypervisor PCI zero-copy API. */
#endif
enum tilepci_err_e {
  /** Largest tilepci error number. */
  TILEPCI_ERR_MAX = -901,

  /** Invalid parameter. */
  TILEPCI_EINVAL = -901,

  /** Bad memory address. */
  TILEPCI_EFAULT = -902,

  /** Process must be bound to a single cpu to invoke tilepci APIs. */
  TILEPCI_EBINDCPU = -903,

  /** Memory region already registered or never registered. */
  TILEPCI_EREGISTERED = -904,

  /** Insufficient command credits to post command. */
  TILEPCI_ECREDITS = -905,

  /** Illegal or unopened channel_id parameter. */
  TILEPCI_ECHANNEL = -906,

  /** Smallest tilepci error number. */
  TILEPCI_ERR_MIN = -906
};

#ifndef __DOXYGEN_TILEPCI_USER_API__

/** The maximum number of PCIE endpoints per chip. */
#define MAX_PCIE_LINKS_PER_CHIP 2

/** The upper four bits of the subsystem ID are used to indicate readiness.
 * @ingroup boot
 */
#define SUBSYSTEM_FLAG_SHIFT 28

/** Different states that can be indicated via the subsystem flag
 * bits.  Note that these are not densely packed for reasons of
 * backwards compatibility.
 *
 * @ingroup boot
 */
enum subsystem_flag_states {
        SUBSYSTEM_FLAG_RESET = 0,   /**< Chip in reset, ready for boot. */
        SUBSYSTEM_FLAG_BOOTED = 4,  /**< Chip is booted, cannot reset.*/
        SUBSYSTEM_FLAG_MINIMAL = 5, /**< iBound supports only reset. */
        SUBSYSTEM_FLAG_PENDING = 6, /**< Chip is booting, driver pending. */
        SUBSYSTEM_FLAG_IBOUND = 8,  /**< iBound running, can reset. */
};

/** Maximum number of huge pages that can be mapped to BAR 1. */
#define PCIE_BAR1_PAGES (1 << (32 - HV_LOG2_PAGE_SIZE_LARGE))

/** BIOS information accessible in the HV device filesystem at
 *  pcie/?/bios, offset TILEPCI_BIOS_OFF.
 * @ingroup tile
 */
struct tlr_bios_info
{
  size_t max_payload_size;   /**< Maximum payload size */
  size_t max_read_size;      /**< Maximum read size */
  uint32_t bus_dev_fn;       /**< Bus/Device/Function */
  int non_posted_credits;    /**< Number of non-posted credits */
  int link_width;            /**< Link width */
  int read_size;             /**< Actual read fragmentation size */
  int read_credits;          /**< Maximum number of outstanding reads. */

  uint64_t bar1_size;        /**< Size of BAR1, 0 if disabled. */
  uint64_t bar1_address;     /**< Bus address of BAR1, 0 if disabled. */
};

/** Offset for reading a tlr_bios_info. */
#define PCIE_BIOS_OFF 0

/** Information about whether a PCI prebooter ran and if so, which
 *  version.  Can be read at pcie/?/bios, offset
 *  TILEPCI_PREBOOTER_OFF. */
union tlr_prebooter_info
{
  struct {
    uint32_t valid:1;        /**< Prebooter ran. */
    uint32_t minor:3;        /**< Minor version number. */
    uint32_t major:4;        /**< Major version number. */
    uint32_t pad:24;         /**< Zero in current API. */
  } bits;                    /**< Bitfield access. */

  uint32_t word;             /**< The whole word. */
};

/** Offset for reading a tlr_prebooter_info. */
#define PCIE_PREBOOTER_OFF 0x100

#endif /* __DOXYGEN_TILEPCI_USER_API__ */

#endif /* __ASSEMBLER__ */

#endif /* _SYS_HV_DRV_PCIE_COMMON_H */
