/*
 * Copyright 2011 Tilera Corporation. All Rights Reserved.
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation, version 2.
 *
 *   This program is distributed in the hope that it will be useful, but
 *   WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, GOOD TITLE or
 *   NON INFRINGEMENT.  See the GNU General Public License for
 *   more details.
 */
#ifndef _HV_IORPC_H_
#define _HV_IORPC_H_

#ifndef __DOXYGEN_API_REF__
/**
 * @file
 * @author griffin
 *
 * Error codes and struct definitions for the IO RPC library.
 *
 * The hypervisor's IO RPC component provides a convenient way for
 * driver authors to proxy system calls between user space, linux, and
 * the hypervisor driver.  The core of the system is a set of Python
 * files that take ".idl" files as input and generates the following
 * source code:
 *
 * - _rpc_call() routines for use in userspace IO libraries.  These
 * routines take an argument list specified in the .idl file, pack the
 * arguments in to a buffer, and read or write that buffer via the
 * Linux iorpc driver.
 *
 * - dispatch_read() and dispatch_write() routines that hypervisor
 * drivers can use to implement most of their dev_pread() and
 * dev_pwrite() methods.  These routines decode the incoming parameter
 * blob, permission check and translate parameters where appropriate,
 * and then invoke a callback routine for whichever RPC call has
 * arrived.  The driver simply implements the set of callback
 * routines.
 *
 * The IO RPC system also includes the Linux 'iorpc' driver, which
 * proxies calls between the userspace library and the hypervisor
 * driver.  The Linux driver is almost entirely device agnostic; it
 * watches for special flags indicating cases where a memory buffer
 * address might need to be translated, etc.  As a result, driver
 * writers can avoid many of the problem cases related to registering
 * hardware resources like memory pages or interrupts.  However, the
 * drivers must be careful to obey the conventions documented below in
 * order to work properly with the generic Linux iorpc driver.
 *
 * @section iorpc_domains Service Domains
 *
 * All iorpc-based drivers must support a notion of service domains.
 * A service domain is basically an application context - state
 * indicating resources that are allocated to that particular app
 * which it may access and (perhaps) other applications may not
 * access.  Drivers can support any number of service domains they
 * choose.  In some cases the design is limited by a number of service
 * domains supported by the IO hardware; in other cases the service
 * domains are a purely software concept and the driver chooses a
 * maximum number of domains based on how much state memory it is
 * willing to preallocate.
 *
 * For example, the mPIPE driver only supports as many service domains
 * as are supported by the mPIPE hardware.  This limitation is
 * required because the hardware implements its own MMIO protection
 * scheme to allow large MMIO mappings while still protecting small
 * register ranges within the page that should only be accessed by the
 * hypervisor.
 *
 * In contrast, drivers with no hardware service domain limitations
 * (for instance the TRIO shim) can implement an arbitrary number of
 * service domains.  In these cases, each service domain is limited to
 * a carefully restricted set of legal MMIO addresses if necessary to
 * keep one application from corrupting another application's state.
 *
 * @section iorpc_conventions System Call Conventions
 *
 * The driver's open routine is responsible for allocating a new
 * service domain for each hv_dev_open() call.  By convention, the
 * return value from open() should be the service domain number on
 * success, or GXIO_ERR_NO_SVC_DOM if no more service domains are
 * available.
 *
 * The implementations of hv_dev_pread() and hv_dev_pwrite() are
 * responsible for validating the devhdl value passed up by the
 * client.  Since the device handle returned by hv_dev_open() should
 * embed the positive service domain number, drivers should make sure
 * that DRV_HDL2BITS(devhdl) is a legal service domain.  If the client
 * passes an illegal service domain number, the routine should return
 * GXIO_ERR_INVAL_SVC_DOM.  Once the service domain number has been
 * validated, the driver can copy to/from the client buffer and call
 * the dispatch_read() or dispatch_write() methods created by the RPC
 * generator.
 *
 * The hv_dev_close() implementation should reset all service domain
 * state and put the service domain back on a free list for
 * reallocation by a future application.  In most cases, this will
 * require executing a hardware reset or drain flow and denying any
 * MMIO regions that were created for the service domain.
 *
 * @section iorpc_data Special Data Types
 *
 * The .idl file syntax allows the creation of syscalls with special
 * parameters that require permission checks or translations as part
 * of the system call path.  Because of limitations in the code
 * generator, APIs are generally limited to just one of these special
 * parameters per system call, and they are sometimes required to be
 * the first or last parameter to the call.  Special parameters
 * include:
 *
 * @subsection iorpc_mem_buffer MEM_BUFFER
 *
 * The MEM_BUFFER() datatype allows user space to "register" memory
 * buffers with a device.  Registering memory accomplishes two tasks:
 * Linux keeps track of all buffers that might be modified by a
 * hardware device, and the hardware device drivers bind registered
 * buffers to particular hardware resources like ingress NotifRings.
 * The MEM_BUFFER() idl syntax can take extra flags like ALIGN_64KB,
 * ALIGN_SELF_SIZE, and FLAGS indicating that memory buffers must have
 * certain alignment or that the user should be able to pass a "memory
 * flags" word specifying attributes like nt_hint or IO cache pinning.
 * The parser will accept multiple MEM_BUFFER() flags.
 *
 * Implementations must obey the following conventions when
 * registering memory buffers via the iorpc flow.  These rules are a
 * result of the Linux driver implementation, which needs to keep
 * track of how many times a particular page has been registered with
 * the hardware so that it can release the page when all those
 * registrations are cleared.
 *
 * - Memory registrations that refer to a resource which has already
 * been bound must return GXIO_ERR_ALREADY_INIT.  Thus, it is an
 * error to register memory twice without resetting (i.e. closing) the
 * resource in between.  This convention keeps the Linux driver from
 * having to track which particular devices a page is bound to.
 *
 * - At present, a memory registration is only cleared when the
 * service domain is reset.  In this case, the Linux driver simply
 * closes the HV device file handle and then decrements the reference
 * counts of all pages that were previously registered with the
 * device.
 *
 * - In the future, we may add a mechanism for unregistering memory.
 * One possible implementation would require that the user specify
 * which buffer is currently registered.  The HV would then verify
 * that that page was actually the one currently mapped and return
 * success or failure to Linux, which would then only decrement the
 * page reference count if the addresses were mapped.  Another scheme
 * might allow Linux to pass a token to the HV to be returned when the
 * resource is unmapped.
 *
 * @subsection iorpc_interrupt INTERRUPT
 *
 * The INTERRUPT .idl datatype allows the client to bind hardware
 * interrupts to a particular combination of IPI parameters - cpu, ipi
 * pl, and event bit number.  This data is passed via a special
 * datatype so that the Linux driver can validate the cpu and pl and
 * the hv generic iorpc code can translate client cpus to real cpus.
 *
 * @subsection iorpc_blob BLOB
 *
 * The BLOB .idl datatype allows the client to write and arbitrary
 * length string of bytes up to the hypervisor driver.  This can be
 * useful for passing up large, arbitrarily structured data like
 * classifier programs.  The iorpc stack takes care of validating the
 * buffer VA and CPA as the data passes up to the hypervisor.  Unlike
 * MEM_BUFFER(), the buffer is not registered - Linux does not bump
 * page refcounts and the HV driver should not reuse the buffer once
 * the system call is complete.
 *
 * @section iorpc_translation Translating User Space Calls
 *
 * The ::iorpc_offset_t structure describes the formatting of the
 * offset that is passed to pread() or pwrite() as part of the
 * generated RPC code.  When the user calls up to Linux, the rpc code
 * fills in all the fields of the offset, including a 16-bit opcode, a
 * 16 bit format indicator, and 32 bits of user-specified
 * "sub-offset".  The opcode indicates which syscall is being
 * requested.  The format indicates whether there is a "prefix struct"
 * at the start of the memory buffer passed to pwrite(), and if so
 * what data is in that prefix struct.  These prefix structs are used
 * to implement MEM_BUFFER() and INTERRUPT datatypes - we arrange to
 * put data that needs translation and permission checks at the start
 * of the buffer so that the Linux driver and generic portions of the
 * HV iorpc code can easily access the data.  The 32 bits of
 * user-specified "sub-offset" are most useful for pread() calls where
 * the user needs to also pass in a few bits indicating which register
 * to read, etc.
 *
 * The Linux iorpc driver watches for system calls that contain prefix
 * structs so that it can translate parameters and bump reference
 * counts as appropriate.  It does not (currently) have any knowledge
 * of the per-device opcodes - it doesn't care what operation you're
 * doing to mPIPE, so long as it can do all the generic book-keeping.
 * The hv/iorpc.h header file defines all of the generic encoding bits
 * needed to translate iorpc calls without knowing which particular
 * opcode is being issued.
 *
 * @section iorpc_globals Global iorpc Calls
 *
 * Implementing mmap() required adding some special iorpc syscalls
 * that are only called by the Linux driver, never by userspace.
 * These include get_mmio_base() and check_mmio_offset().  These
 * routines are described in globals.idl and must be included in every
 * iorpc driver.  By providing these routines in every driver, Linux's
 * mmap implementation can easily get the PTE bits it needs and
 * validate the PA offset without needing to know the per-device
 * opcodes to perform those tasks.
 *
 * @section iorpc_kernel Supporting gxio APIs in the Kernel
 *
 * The iorpc code generator also supports generation of kernel code
 * implementing the gxio APIs.  This capability is currently used by
 * the mPIPE network driver, and will likely be used by the TRIO root
 * complex and endpoint drivers and perhaps an in-kernel crypto
 * driver.  Each driver that wants to instantiate iorpc calls in the
 * kernel needs to generate a kernel version of the generate rpc code
 * and (probably) copy any related gxio source files into the kernel.
 * The mPIPE driver provides a good example of this pattern.
 *
 */

#ifdef __KERNEL__
#include <linux/stddef.h>
#else
#include <stddef.h>
#endif

#if defined(__HV__)
#include <hv/hypervisor.h>
#elif defined(__KERNEL__)
#include "hypervisor.h"
#include <linux/types.h>
#else
#include <stdint.h>
#endif


/** Code indicating translation services required within the RPC path.
 * These indicate whether there is a translatable struct at the start
 * of the RPC buffer and what information that struct contains.
 */
enum iorpc_format_e
{
  /** No translation required, no prefix struct. */
  IORPC_FORMAT_NONE,

  /** Prefix struct contains user VA and size. */
  IORPC_FORMAT_USER_MEM,

  /** Prefix struct contains CPA, size, and homing bits. */
  IORPC_FORMAT_KERNEL_MEM,

  /** Prefix struct contains interrupt. */
  IORPC_FORMAT_INTERRUPT,
};


/** Generate an opcode given format and code. */
#define IORPC_OPCODE(FORMAT, CODE) (((FORMAT) << 16) | (CODE))


/** The offset passed through the read() and write() system calls
    combines an opcode with 32 bits of user-specified offset. */
typedef union
{
  uint64_t offset;              /**< All bits. */

  struct
  {
    uint16_t code;              /**< RPC code. */
    uint16_t format;            /**< iorpc_format_e */
    uint32_t sub_offset;        /**< caller-specified offset. */
  };

  uint32_t opcode;              /**< Opcode combines code & format. */
}
iorpc_offset_t;


/** Homing and cache hinting bits that can be used by IO devices. */
typedef struct
{
  unsigned long lotar_x:4;      /**< lotar X bits (or Gx page_mask). */
  unsigned long lotar_y:4;      /**< lotar Y bits (or Gx page_offset). */
  unsigned long hfh:1;          /**< Uses hash-for-home. */
  unsigned long nt_hint:1;      /**< Non-temporal hint. */
  unsigned long io_pin:1;       /**< Only fill 'IO' cache ways. */
}
iorpc_mem_attr_t;

/** Set the nt_hint bit. */
#define IORPC_MEM_BUFFER_FLAG_NT_HINT (1 << 0)

/** Set the IO pin bit. */
#define IORPC_MEM_BUFFER_FLAG_IO_PIN (1 << 1)


/** A structure used to describe memory registration.  Different
    protection levels describe memory differently, so this union
    contains all the different possible descriptions.  As a request
    moves up the call chain, each layer translates from one
    description format to the next.  In particular, the Linux iorpc
    driver translates user VAs into CPAs and homing parameters. */
typedef union
{
  struct
  {
    void* va;                   /**< User virtual address. */
    size_t size;                /**< Buffer size. */
    unsigned long flags;        /**< nt_hint, IO pin. */
  }
  user;                         /**< Buffer as described by user apps. */

  struct
  {
    unsigned long long cpa;     /**< Client physical address. */
    size_t size;                /**< Buffer size. */
#if defined(__KERNEL__) || defined(__HV__)
    HV_PTE pte;                 /**< PTE describing memory homing. */
#else
    uint64_t pte;
#endif
    unsigned long flags;        /**< nt_hint, IO pin. */
  }
  kernel;                       /**< Buffer as described by kernel. */

  struct
  {
    unsigned long long pa;      /**< Physical address. */
    size_t size;                /**< Buffer size. */
    iorpc_mem_attr_t attr;      /**< Homing and locality hint bits. */
  }
  hv;                           /**< Buffer parameters for HV driver. */
} iorpc_mem_buffer_t;

/** Info about an interrupt. */
typedef struct
{
  int x; /**< X coord. */
  int y; /**< Y coord. */
  int i; /**< int_num */
  int e; /**< evt_num */
} iorpc_interrupt_t;

#endif /* ! __DOXYGEN_API_REF__ */

#ifdef __DOXYGEN_API_REF__
/**
 * @addtogroup gxio_error
 * @{
 */

/** Error codes returned by gxio_ API calls. */
#else
/** The various iorpc devices use error codes from -1100 to -1299.
 *
 * This range is distinct from netio (-700 to -799), the hypervisor
 * (-800 to -899), tilepci (-900 to -999), ilib (-1000 to -1099),
 * and gxcr (-1300 to -1399).
 */
#endif
enum gxio_err_e {

#ifndef __DOXYGEN_API_REF__
  /** Largest iorpc error number. */
  GXIO_ERR_MAX = -1101,
#endif


  /********************************************************/
  /*                   Generic Error Codes                */
  /********************************************************/

  /** Bad RPC opcode - possible version incompatibility. */
  GXIO_ERR_OPCODE = -1101,

  /** Invalid parameter. */
  GXIO_ERR_INVAL = -1102,

  /** Memory buffer did not meet alignment requirements. */
  GXIO_ERR_ALIGNMENT = -1103,

  /** Memory buffers must be coherent and cacheable. */
  GXIO_ERR_COHERENCE = -1104,

  /** Resource already initialized. */
  GXIO_ERR_ALREADY_INIT = -1105,

  /** No service domains available. */
  GXIO_ERR_NO_SVC_DOM = -1106,

  /** Illegal service domain number. */
  GXIO_ERR_INVAL_SVC_DOM = -1107,

  /** Illegal MMIO address. */
  GXIO_ERR_MMIO_ADDRESS = -1108,

  /** Illegal interrupt binding. */
  GXIO_ERR_INTERRUPT = -1109,

  /** Unreasonable client memory. */
  GXIO_ERR_CLIENT_MEMORY = -1110,
  
  /** No more IOTLB entries. */
  GXIO_ERR_IOTLB_ENTRY = -1111,

  /** Invalid memory size. */
  GXIO_ERR_INVAL_MEMORY_SIZE = -1112,

  /** Insufficient DMA credits. */
  GXIO_ERR_DMA_CREDITS = -1150,
  

#ifndef __DOXYGEN_API_REF__

  /********************************************************/
  /*                 Test Device Error Codes              */
  /********************************************************/

  /** Illegal register number. */
  GXIO_TEST_ERR_REG_NUMBER = -1120,

  /** Illegal buffer slot. */
  GXIO_TEST_ERR_BUFFER_SLOT = -1121,

#endif

  /********************************************************/
  /*                    MPIPE Error Codes                 */
  /********************************************************/


  /** Invalid buffer size. */
  GXIO_MPIPE_ERR_INVAL_BUFFER_SIZE = -1131,

  /** Cannot allocate buffer stack. */
  GXIO_MPIPE_ERR_NO_BUFFER_STACK = -1140,

  /** Invalid buffer stack number. */
  GXIO_MPIPE_ERR_BAD_BUFFER_STACK = -1141,

  /** Cannot allocate NotifRing. */
  GXIO_MPIPE_ERR_NO_NOTIF_RING = -1142,

  /** Invalid NotifRing number. */
  GXIO_MPIPE_ERR_BAD_NOTIF_RING = -1143,

  /** Cannot allocate NotifGroup. */
  GXIO_MPIPE_ERR_NO_NOTIF_GROUP = -1144,

  /** Invalid NotifGroup number. */
  GXIO_MPIPE_ERR_BAD_NOTIF_GROUP = -1145,

  /** Cannot allocate bucket. */
  GXIO_MPIPE_ERR_NO_BUCKET = -1146,

  /** Invalid bucket number. */
  GXIO_MPIPE_ERR_BAD_BUCKET = -1147,

  /** Cannot allocate eDMA ring. */
  GXIO_MPIPE_ERR_NO_EDMA_RING = -1148,

  /** Invalid eDMA ring number. */
  GXIO_MPIPE_ERR_BAD_EDMA_RING = -1149,

  /* ERR_EDMA_CREDITS and ERR_IOTLB_ENTRY used to be here. */
  
  /** Empty iqueue. */
  GXIO_MPIPE_ERR_IQUEUE_EMPTY = -1152,

  /** Empty rules. */
  GXIO_MPIPE_ERR_RULES_EMPTY = -1160,

  /** Full rules. */
  GXIO_MPIPE_ERR_RULES_FULL = -1161,

  /** Corrupt rules. */
  GXIO_MPIPE_ERR_RULES_CORRUPT = -1162,

  /** Invalid rules. */
  GXIO_MPIPE_ERR_RULES_INVALID = -1163,

  /** Classifier is too big. */
  GXIO_MPIPE_ERR_CLASSIFIER_TOO_BIG = -1170,

  /** Classifier is too complex. */
  GXIO_MPIPE_ERR_CLASSIFIER_TOO_COMPLEX = -1171,

  /** Classifier has bad header. */
  GXIO_MPIPE_ERR_CLASSIFIER_BAD_HEADER = -1172,

  /** Classifier has bad contents. */
  GXIO_MPIPE_ERR_CLASSIFIER_BAD_CONTENTS = -1173,

  /** Classifier encountered invalid symbol. */
  GXIO_MPIPE_ERR_CLASSIFIER_INVAL_SYMBOL = -1174,

  /** Classifier encountered invalid bounds. */
  GXIO_MPIPE_ERR_CLASSIFIER_INVAL_BOUNDS = -1175,

  /** Classifier encountered invalid relocation. */
  GXIO_MPIPE_ERR_CLASSIFIER_INVAL_RELOCATION = -1176,

  /** Classifier encountered undefined symbol. */
  GXIO_MPIPE_ERR_CLASSIFIER_UNDEF_SYMBOL = -1177,


  /********************************************************/
  /*                    TRIO  Error Codes                 */
  /********************************************************/

  /** Cannot allocate memory map region. */
  GXIO_TRIO_ERR_NO_MEMORY_MAP = -1180,

  /** Invalid memory map region number. */
  GXIO_TRIO_ERR_BAD_MEMORY_MAP = -1181,

  /** Cannot allocate scatter queue. */
  GXIO_TRIO_ERR_NO_SCATTER_QUEUE = -1182,

  /** Invalid scatter queue number. */
  GXIO_TRIO_ERR_BAD_SCATTER_QUEUE = -1183,

  /** Cannot allocate push DMA ring. */
  GXIO_TRIO_ERR_NO_PUSH_DMA_RING = -1184,

  /** Invalid push DMA ring index. */
  GXIO_TRIO_ERR_BAD_PUSH_DMA_RING = -1185,

  /** Cannot allocate pull DMA ring. */
  GXIO_TRIO_ERR_NO_PULL_DMA_RING = -1186,

  /** Invalid pull DMA ring index. */
  GXIO_TRIO_ERR_BAD_PULL_DMA_RING = -1187,

  /** Cannot allocate PIO region. */
  GXIO_TRIO_ERR_NO_PIO = -1188,

  /** Invalid PIO region index. */
  GXIO_TRIO_ERR_BAD_PIO = -1189,

  /** Cannot allocate ASID. */
  GXIO_TRIO_ERR_NO_ASID = -1190,

  /** Invalid ASID. */
  GXIO_TRIO_ERR_BAD_ASID = -1191,


  /********************************************************/
  /*                    MICA Error Codes                  */
  /********************************************************/

  /** No such accelerator type. */
  GXIO_MICA_ERR_BAD_ACCEL_TYPE = -1220,

  /** Cannot allocate context. */
  GXIO_MICA_ERR_NO_CONTEXT = -1221,

#ifndef __DOXYGEN_API_REF__
  /** Smallest iorpc error number. */
  GXIO_ERR_MIN = -1299
#endif
};

#ifdef __DOXYGEN_API_REF__
/** @} */
#endif

#endif /* !_HV_IORPC_H_ */
