/*
 * Copyright 2011 Tilera Corporation. All Rights Reserved.
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation, version 2.
 *
 *   This program is distributed in the hope that it will be useful, but
 *   WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, GOOD TITLE or
 *   NON INFRINGEMENT.  See the GNU General Public License for
 *   more details.
 */

#include <linux/spinlock.h>
#include <linux/hugetlb.h>
#include <linux/highmem.h>
#include <linux/bug.h>
#include <asm/pgtable.h>
#include <asm/homecache.h>
#include <hv/hypervisor.h>

static spinlock_t map_lock = __SPIN_LOCK_UNLOCKED(&map_lock);
struct page *map[CONFIG_NR_HUGE_VMAPS];

void *huge_vmap(struct page *page, pgprot_t prot)
{
	int i;
	unsigned long flags, addr;
	pgd_t *pgd;
	pmd_t *pmd;

	spin_lock_irqsave(&map_lock, flags);
	for (i = 0; i < CONFIG_NR_HUGE_VMAPS; ++i) {
		if (map[i] == NULL) {
			map[i] = page;
			break;
		}
	}
	spin_unlock_irqrestore(&map_lock, flags);

	if (i == CONFIG_NR_HUGE_VMAPS)
		return NULL;
	addr = HUGE_VMAP_BASE + (i * HPAGE_SIZE);
	pgd = swapper_pg_dir + pgd_index(addr);
	pmd = pmd_offset(pud_offset(pgd, addr), addr);
	set_pte_order((pte_t *)pmd,
		      pte_mkhuge(pfn_pte(page_to_pfn(page), prot)),
		      HUGETLB_PAGE_ORDER);
	return (void *) addr;
}

/*
 * Rather than trying to do sophisticated TLB-management games like
 * the pkmap code, we just do a global flush when we unmap.
 * The assumption is that huge pages are acquired long-term by
 * kernel drivers like the gbe and xgbe.
 */
void huge_vunmap(void *addr)
{
	int i;
	unsigned long flags;

	BUG_ON((unsigned long) addr & (HPAGE_SIZE-1));
	i = ((unsigned long) addr - HUGE_VMAP_BASE) >> HPAGE_SHIFT;
	BUG_ON(i >= CONFIG_NR_HUGE_VMAPS);

	spin_lock_irqsave(&map_lock, flags);
	map[i] = NULL;
	spin_unlock_irqrestore(&map_lock, flags);

	flush_remote(0, 0, NULL, (HV_VirtAddr) addr, HPAGE_SIZE, HPAGE_SIZE,
		     cpu_online_mask, NULL, 0);
}
